<?php

namespace IZON\Utils;

use function mb_strrpos;
use function mb_substr;

/**
 * utility functions for strings
 * @package IZON\Utils
 */
class StringUtils
{
    /**
     * true if $haystack starts with $needle
     * @param string $haystack
     * @param string $needle
     * @return bool
     */
    public static function startsWith(string $haystack, string $needle): bool
    {
        return str_starts_with($haystack, $needle);
    }

    /**
     * true if $haystack starts with $needle
     * @param string $haystack
     * @param string $needle
     * @return bool
     */
    public static function endsWith(string $haystack, string $needle): bool
    {
        return str_ends_with($haystack, $needle);
    }

    /**
     * true id $haystack begins with any string from $needles
     * @param string $haystack string
     * @param string[] $needles array of string patterns
     *
     * @return bool
     */
    public static function startWithAny(string $haystack, array $needles): bool
    {
        foreach ($needles as $pattern) {
            if (str_starts_with($haystack, $pattern)) {
                return true;
            }
        }
        return false;
    }

    /**
     * true id $haystack ends with any string from $needles
     * @param string $haystack string
     * @param string[] $needles array of string patterns
     *
     * @return bool
     */
    public static function endsWithAny(string $haystack, array $needles): bool
    {
        foreach ($needles as $pattern) {
            if (str_ends_with($haystack, $pattern)) {
                return true;
            }
        }
        return false;
    }

    /**
     * function safety truncates (keeps complete words) string to be shorter than $length and add suffix
     *
     * @param string $string string to truncate
     * @param int $length max length of sting + $truncateSuffix
     * @param string $truncateSuffix default '...'
     * @return string
     * @throws \Exception
     */
    public static function safeTruncate(string $string, int $length, string $truncateSuffix = '...'): string
    {
        if ($length < 1) {
            throw new \Exception('parameter length must be grater then 0');
        }
        if (mb_strlen($string) <= $length) {
            return $string;
        }

        $truncateSuffixLength = mb_strlen($truncateSuffix);
        $truncated = $string;
        while (mb_strlen($truncated) + $truncateSuffixLength > $length) {
            $lastWordPos = max(
                mb_strrpos($truncated, ' ', -1), // find last empty space
                mb_strrpos($truncated, "\n", -1), // find last new line
                mb_strrpos($truncated, "\t", -1), // find last new tab
            );
            if ($lastWordPos === false) { // space not found
                break;
            }
            $truncated = mb_substr($truncated, 0, $lastWordPos);
        }

        return $truncated . $truncateSuffix;
    }

    /**
     * @deprecated use splitToUniqueParts
     * @param string $string
     * @return string[]
     */
    public static function stringToUniqueParts(string $string): array
    {
        return self::splitToUniqueParts($string, fn ($part) => "%{$part}%");
    }

    /**
     * @param string $string string to split and modify
     * @param callable|null $partModifier function to modify each unique part
     * @param string $splitPattern regex pattern to split string
     * @return string[]
     * @throws \InvalidArgumentException if string split fails
     */
    public static function splitToUniqueParts(
        string $string,
        ?callable $partModifier = null,
        string $splitPattern = '#\s+#',
    ): array {
        if ($partModifier === null) {
            $partModifier = fn ($part): string => $part;
        }

        $splitted = preg_split($splitPattern, $string);
        if ($splitted === false) {
            throw new \InvalidArgumentException('string split failed. $splitPattern is probably invalid.');
        }
        $slittedArray = array_filter($splitted);
        $modifiedArray = array_map(
            $partModifier,
            $slittedArray,
        );
        $uniqueArray = array_unique($modifiedArray);
        return array_values($uniqueArray);
    }
}
