<?php


namespace IZON\Utils;

use Exception;
use InvalidArgumentException;
use ReflectionClass;
use ReflectionMethod;

/**
 * utility functions for objects
 * @package IZON\Utils
 */
class ObjectUtils
{
    /**
     * creates shallow array from object, does not convert object properties itself to array
     * uses getters and is methods
     * @param object $object
     * @param array<string> $ignoreProperties
     * @return array<string, mixed>
     */
    public static function toArray(object $object, array $ignoreProperties = []): array
    {
        $refClass = new ReflectionClass($object);

        $array = [];
        foreach ($refClass->getMethods(ReflectionMethod::IS_PUBLIC) as $reflectionMethod) {
            $methodName = $reflectionMethod->getName();
            if (
                !str_starts_with($methodName, "get")
                && !str_starts_with($methodName, "is")
                && $reflectionMethod->getNumberOfParameters() > 0
            ) {
                continue;
            }

            $propertyName = self::createPropertyName($methodName);
            if (!in_array($propertyName, $ignoreProperties)) { // property is not ignored
                $array[$propertyName] = $object->$methodName();
            }
        }

        return $array;
    }

    /**
     * copies values from $sourceArray to $targetObject using setters
     * @param object $targetObject
     * @param array<string, mixed> $sourceArray
     * @param array<string> $ignoreProperties
     */
    public static function fromArray(object $targetObject, array $sourceArray, array $ignoreProperties = []): object
    {
        $refClass = new ReflectionClass($targetObject);

        foreach ($sourceArray as $propertyName => $property) {
            if (in_array($propertyName, $ignoreProperties)) { // property is ignored
                continue;
            }

            $setMethodName = self::createSetterMethodName($propertyName);
            if (
                $refClass->hasMethod($setMethodName) // set method exists
                && $refClass->getMethod($setMethodName)->getNumberOfParameters() === 1 // set method has one parameter
                && $refClass->getMethod($setMethodName)->isPublic() // set method is public
            ) {
                $targetObject->$setMethodName($property);
            }
        }

        return $targetObject;
    }

    /**
     * provadi shalow copy
     * kopiruje property z $orig do $dest pres gettery a settery
     * property se kopiruje pokud existuje get* metoda v $orig a set* metoda v $dest
     * @param object $dest to
     * @param object $orig from
     * @param array<string> $ignoreProperties properties to be ignored while copying
     */
    public static function copyProperties(object $dest, object $orig, array $ignoreProperties = []): void
    {
        $originRefClass = new ReflectionClass($orig);
        $destRefClass = new ReflectionClass($dest);

        foreach ($originRefClass->getMethods(ReflectionMethod::IS_PUBLIC) as $originRefMethod) {
            $methodName = $originRefMethod->getName();
            if (
                !str_starts_with($methodName, "get")
                && !str_starts_with($methodName, "is")
                && $originRefMethod->getNumberOfParameters() > 0
            ) {
                continue;
            }

            $propertyName = self::createPropertyName($methodName);
            if (in_array($propertyName, $ignoreProperties)) { // property is ignored
                continue;
            }

            $setMethodName = self::createSetterMethodName($propertyName);

            if (
                $destRefClass->hasMethod($setMethodName) // set method exists
                && $destRefClass->getMethod($setMethodName)->getNumberOfParameters() === 1 // set method has one parameter
                && $destRefClass->getMethod($setMethodName)->isPublic() // set method is public
            ) {
                $val = $orig->$methodName();
                $dest->$setMethodName($val);
            }
        }
    }

    /**
     * gets property value from object using getter or is method
     * @param object $object
     * @param string $property
     * @return mixed
     * @throws Exception
     */
    public static function getPropertyValue(object $object, string $property): mixed
    {
        $getterName = self::createGetterMethodName($property);
        if (method_exists($object, $getterName)) {
            return $object->$getterName();
        }

        $istter = self::createIsMethodName($property);
        if (method_exists($object, $istter)) {
            return $object->$istter();
        }

        throw new Exception("Nor getter $getterName nor is method $istter exists.");
    }

    /**
     * sets property value to object using setter method
     * @param object $object
     * @param string $property
     * @param mixed $value
     * @return void
     * @throws Exception
     */
    public static function setPropertyValue(object $object, string $property, mixed $value): void
    {
        $setter = self::createSetterMethodName($property);
        if (method_exists($object, $setter)) {
            $object->$setter($value);
            return;
        }

        throw new Exception("Setter method $setter  does not exist.");
    }

    /**
     * @param string $propertyName
     * @return string
     */
    public static function createGetterMethodName(string $propertyName): string
    {
        $methodName = "get" . ucfirst($propertyName);
        return $methodName;
    }

    /**
     * creates is method name from property name
     * @param string $propertyName
     * @return string
     */
    public static function createIsMethodName(string $propertyName): string
    {
        $methodName = "is" . ucfirst($propertyName);
        return $methodName;
    }

    /**
     * creates setter method name from property name
     * @param string $propertyName
     * @return string
     */
    public static function createSetterMethodName(string $propertyName): string
    {
        $methodName = "set" . ucfirst($propertyName);
        return $methodName;
    }

    /**
     * creates property name from getter, setter or is method name
     * @param string $methodName
     * @return string
     */
    public static function createPropertyName(string $methodName): string
    {
        if (str_starts_with($methodName, "get")) {
            $propertyName = mb_substr($methodName, 3);
        } elseif (str_starts_with($methodName, "is")) {
            $propertyName = mb_substr($methodName, 2);
        } elseif (str_starts_with($methodName, "set")) {
            $propertyName = mb_substr($methodName, 3);
        } else {
            throw new InvalidArgumentException("Method $methodName is not getter, setter or is method.");
        }

        return mb_strtolower(mb_substr($propertyName, 0, 1)) . mb_substr($propertyName, 1);
    }
}
