<?php

namespace IZON\IO;

use IZON\Exceptions\NotImplementedException;
use IZON\Exceptions\UnableToWriteFileException;
use RuntimeException;
use Serializable;

/**
 * temp file that exists only during script run
 */
class TempFile extends AbstractFile implements FileInterface, Serializable
{
    protected string $fileName;

    /**
     * @var resource
     */
    protected $tmpResource;

    protected string $path;

    protected ?string $mimeType = null;

    /**
     * @param string $fileName
     */
    public function __construct(string $fileName)
    {
        $this->fileName = $fileName;
        $tmpResource = tmpfile();
        if ($tmpResource === false) {
            throw new RuntimeException("Could not create temporary file.");
        }
        $this->tmpResource = $tmpResource;
        $this->path = stream_get_meta_data($this->tmpResource)['uri'];
    }


    /**
     * {@inheritDoc}
     */
    public function getFileContents(): string
    {
        rewind($this->tmpResource);
        $size = $this->getFileSize();
        if ($size === 0) {
            return '';
        }
        $fileContents = fread($this->tmpResource, $size);
        if ($fileContents === false) {
            throw new RuntimeException("Could not read file contents.");
        }
        return $fileContents;
    }

    /**
     * {@inheritDoc}
     */
    public function getFileExtension(): string
    {
        return pathinfo($this->fileName, PATHINFO_EXTENSION);
    }

    /**
     * {@inheritDoc}
     */
    public function getFileName(): string
    {
        return $this->fileName;
    }

    /**
     * {@inheritDoc}
     */
    public function getFileSize(): int
    {
        $stat = fstat($this->tmpResource);
        if ($stat === false) {
            throw new RuntimeException("Could not get file size.");
        }
        /** @var int<0, max> $size */
        $size = $stat['size'];
        return $size;
    }

    /**
     * {@inheritDoc}
     */
    public function getFileStream()
    {
        return $this->tmpResource;
    }

    /**
     * {@inheritDoc}
     */
    public function getMimeType(): string
    {
        if (!is_null($this->mimeType)) {
            return $this->mimeType;
        }
        $mime = mime_content_type($this->tmpResource);
        if ($mime === false) {
            throw new NotImplementedException("Fallback when mime is not guessable");
        }
        $this->mimeType = $mime;
        return $mime;
    }

    /**
     * {@inheritDoc}
     */
    public function getTimestamp(): int
    {
        $stat = fstat($this->tmpResource);
        if ($stat === false) {
            throw new RuntimeException("Could not get file timestamp.");
        }
        return $stat['mtime'];
    }

    /**
     * https://www.php.net/manual/en/splfileobject.fwrite.php
     * @param string $data
     * @param null|int<0, max> $length
     * @return int
     * @throws UnableToWriteFileException
     */
    public function write(string $data, ?int $length = null): int
    {
        /** @phpstan-ignore-next-line check if not negative */
        if ($length < 0) {
            throw new UnableToWriteFileException('Length must be greater than or equal to 0');
        }
        $fileSize = fwrite($this->tmpResource, $data, $length);
        if ($fileSize === false) {
            throw new UnableToWriteFileException('Unable to write to tmp file');
        }
        return $this->getFileSize();
    }

    public function getOriginUID(): ?string
    {
        return null;
    }

    public function getFileUID(): string
    {
        return $this->path;
    }

    /**
     * {@inheritDoc}
     */
    public function serialize()
    {
        throw new NotImplementedException();
    }

    /**
     * {@inheritDoc}
     */
    public function unserialize($serialized)
    {
        throw new NotImplementedException();
    }

    /**
     * @return array<string, mixed>
     * @throws NotImplementedException
     */
    public function __serialize(): array
    {
        throw new NotImplementedException();
    }

    /**
     * @param array<string, mixed> $data
     * @return void
     * @throws NotImplementedException
     */
    public function __unserialize(array $data): void
    {
        throw new NotImplementedException();
    }
}
