<?php

namespace IZON\Utils;

use function mb_strrpos;
use function mb_substr;

/**
 * utility functions for strings
 * @package IZON\Utils
 */
class StringUtils
{
    /**
     * true if $haystack starts with $needle
     * @param string $haystack
     * @param string $needle
     * @return bool
     */
    public static function startsWith(string $haystack, string $needle): bool
    {
        return str_starts_with($haystack, $needle);
    }

    /**
     * true if $haystack starts with $needle
     * @param string $haystack
     * @param string $needle
     * @return bool
     */
    public static function endsWith(string $haystack, string $needle): bool
    {
        return str_ends_with($haystack, $needle);
    }

    /**
     * true id $haystack begins with any string from $needles
     * @param string $haystack string
     * @param string[] $needles array of string patterns
     *
     * @return bool
     */
    public static function startWithAny(string $haystack, array $needles): bool
    {
        foreach ($needles as $pattern) {
            if (str_starts_with($haystack, $pattern)) {
                return true;
            }
        }
        return false;
    }

    /**
     * true id $haystack ends with any string from $needles
     * @param string $haystack string
     * @param string[] $needles array of string patterns
     *
     * @return bool
     */
    public static function endsWithAny(string $haystack, array $needles): bool
    {
        foreach ($needles as $pattern) {
            if (str_ends_with($haystack, $pattern)) {
                return true;
            }
        }
        return false;
    }

    /**
     * function safety truncates (keeps complete words) string to be shorter than $length and add suffix
     *
     * @param string $string string to truncate
     * @param int $length max length of sting + $truncateSuffix
     * @param string $truncateSuffix default '...'
     * @return string
     * @throws \Exception
     */
    public static function safeTruncate(string $string, int $length, string $truncateSuffix = '...'): string
    {
        if ($length < 1) {
            throw new \Exception('parameter length must be grater then 0');
        }
        if (mb_strlen($string) <= $length) {
            return $string;
        }

        $trucatedSuffixLength = mb_strlen($truncateSuffix);
        $string = mb_substr($string, 0, $length); // střihni
        $newLength = 0;

        do {
            $lastWordPos = mb_strrpos($string, ' ', -1); // find last space
            if ($lastWordPos === false) { // space not found
                break;
            }
            $string = mb_substr($string, 0, $lastWordPos);
            $newLength = mb_strlen($string) + $trucatedSuffixLength;
        } while ($newLength > $length);

        return $string . $truncateSuffix;
    }
}
