<?php

namespace IZON\Utils;

use InvalidArgumentException;

class PasswordUtils
{
    public const MIN_PASSWORD_LENGTH = 6;

    /**
     * use numeric characters
     */
    public const NUMERIC = 1;

    /**
     * use upper case characters
     */
    public const UPPER_CASE = 2;

    /**
     * use upper lower characters
     */
    public const LOWER_CASE = 4;

    private const NUMERIC_CHARS = '0123456789';

    private const UPPER_CASE_CHARS = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';

    private const LOWER_CASE_CHARS = 'abcdefghijklmnopqrstuvwxyz';

    /**
     * @param int $length length of generated password
     * @param int $allowedCharsFlag flags of allowed chars
     * @return string
     */
    public static function generatePassword(
        int $length = 16,
        int $allowedCharsFlag = self::NUMERIC | self::UPPER_CASE | self::LOWER_CASE
    ): string {
        if ($length < self::MIN_PASSWORD_LENGTH) {
            throw new InvalidArgumentException('Password length must be at least ' . self::MIN_PASSWORD_LENGTH);
        }

        $sourceChars = '';
        $charSetsCount = 0;
        $passwordString = '';

        if ($allowedCharsFlag & self::NUMERIC) {
            $sourceChars .= self::NUMERIC_CHARS;
            $charSetsCount++;
            // add at least one numeric char
            $charIndex = random_int(0, strlen(self::NUMERIC_CHARS) - 1);
            $passwordString .= substr(self::NUMERIC_CHARS, $charIndex, 1);
        }
        if ($allowedCharsFlag & self::UPPER_CASE) {
            $sourceChars .= self::UPPER_CASE_CHARS;
            $charSetsCount++;
            // add at least one upper case char
            $charIndex = random_int(0, strlen(self::UPPER_CASE_CHARS) - 1);
            $passwordString .= substr(self::UPPER_CASE_CHARS, $charIndex, 1);
        }
        if ($allowedCharsFlag & self::LOWER_CASE) {
            $sourceChars .= self::LOWER_CASE_CHARS;
            $charSetsCount++;
            // add at least one lower case char
            $charIndex = random_int(0, strlen(self::LOWER_CASE_CHARS) - 1);
            $passwordString .= substr(self::LOWER_CASE_CHARS, $charIndex, 1);
        }

        if ($charSetsCount == 0) {
            throw new InvalidArgumentException('At least one char set must be allowed');
        }

        $sourceCharsCount = strlen($sourceChars);
        assert($sourceCharsCount > 1);

        // add rest of chars random chars
        for ($i = 0; $i < $length - $charSetsCount; $i++) {
            $passwordString .= substr($sourceChars, random_int(0, $sourceCharsCount - 1), 1);
        }

        // shuffle password string
        return str_shuffle($passwordString);
    }
}
