<?php


namespace IZON\Utils;

use IZON\IO\Exceptions\FileNotFoundException;
use IZON\IO\Exceptions\FilesystemAccessException;
use IZON\IO\FileInterface;
use Exception;
use TypeError;

/**
 * utility functions for files
 * @package IZON\Utils
 */
class FileUtils
{
    /**
     * normalizes $fileName to OS independent format
     * form windows changes \ to / and if uppercase chars are present converts them to lowercase
     * @param string $fileName
     * @return string
     */
    public static function normalizeFileName(string $fileName): string
    {
        if (str_starts_with(strtolower(PHP_OS), "win")) { // is some kind of windows
            $fileName = str_replace("\\", "/", $fileName);
            $fileName = mb_strtolower($fileName);
        }
        return $fileName;
    }

    /**
     * returns true if $file is directly in $dirName or in some of its subdirectories
     * @param string $fileName
     * @param string $dirName
     * @return bool
     */
    public static function isInSubDir(string $fileName, string $dirName): bool
    {
        $fileName = self::normalizeFileName($fileName);
        $directoryName = self::normalizeFileName($dirName);
        return str_starts_with($fileName, $directoryName);
    }


    /**
     * Deletes file or directory with all its contents
     * @param string $filePath
     * @throw FileNotFoundException if file does not exist anymore
     * @throw Exception if there is some error accessing filesystem
     */
    public static function delete(string $filePath): void
    {
        if (!file_exists($filePath)) {
            throw new FileNotFoundException('File "' . $filePath . '" does not exist.');
        }

        if (!is_dir($filePath)) { // is not directory
            unlink($filePath);
        }

        $dirContent = scandir($filePath);
        if ($dirContent === false) {
            throw new FilesystemAccessException('Error reading directory "' . $filePath . '" contents.');
        }
        // delete contents of directory
        foreach ($dirContent as $subFile) {
            $absoluteFileDir = $filePath . '/' . $subFile;

            if (is_dir($absoluteFileDir)) {
                if (
                    $subFile != "."
                    && $subFile != ".."
                ) {
                    self::delete($absoluteFileDir);
                }
            } else {
                unlink($absoluteFileDir);
            }
        }

        rmdir($filePath);
    }

    /**
     * @var array<string>|null mime types of raster images
     */
    protected const RASTER_IMAGE_MIME_TYPES = [
        'application/x-shockwave-flash',
        'image/bmp',
        'image/gif',
        'image/iff',
        'image/jp2',
        'image/jpeg',
        'image/png',
        'image/psd',
        'image/tiff',
        'image/vnd.microsoft.icon',
        'image/vnd.wap.wbmp',
        'image/xbm',
        'image/webp',
    ];

    /**
     * @return array<string> known mime types of raster images
     */
    protected static function getRasterImageMimeTypes(): array
    {
        return self::RASTER_IMAGE_MIME_TYPES;
    }

    /**
     * @param string $mimeType
     * @return bool true if $mimeType represents raster image
     */
    protected static function isRasterImageMimeType(string $mimeType): bool
    {
        return in_array($mimeType, self::RASTER_IMAGE_MIME_TYPES);
    }

    /**
     * @param string|FileInterface $file file to test if is raster image
     * @return bool
     * @throw TypeError if $file is not string or FileInterface
     * @throw Exception if unable to get mimetype of file
     */
    public static function isRasterImage($file): bool
    {
        if ($file instanceof FileInterface) {
            $mimeType = $file->getMimeType();
            return self::isRasterImageMimeType($mimeType);
        } elseif (is_string($file)) {
            $mimeType = mime_content_type($file);
            if ($mimeType === false) {
                throw new Exception('Unable get mimetype of "' . $file . '".');
            }
            return self::isRasterImageMimeType($mimeType);
        }
        // @phpstan-ignore-next-line
        throw new TypeError('Only FileInterface and file name string are supported.');
    }

    /**
     * @param int $bytes filesize in bytes
     * @param int $decimals number of decimal places do show
     * @return string human-readable size (e.g. 42MB)
     */
    public static function getHumanReadableFileSize(int $bytes, int $decimals = 2): string
    {
        $size = ['B', 'kB', 'MB', 'GB', 'TB', 'PB', 'EB', 'ZB', 'YB'];
        $factor = floor((strlen((string)$bytes) - 1) / 3);

        return sprintf("%.{$decimals}f", $bytes / pow(1024, $factor)) . @$size[$factor];
    }
}
