<?php

namespace IZON\Utils\Tests;

use IZON\Utils\StringUtils;
use PHPUnit\Framework\TestCase;

class StringUtilsTest extends TestCase
{
    public function testStartsWith(): void
    {
        $haystack = 'blah, bleh bluh';
        foreach (['bl', 'blah', 'blah,', 'blah, b'] as $needle) {
            $this->assertTrue(StringUtils::startsWith($haystack, $needle));
        }
    }

    public function testStartsWithWrong(): void
    {
        $haystack = 'blah, bleh bluh';
        foreach (['blh', 'bla,', 'blah,b'] as $needle) {
            $this->assertTrue(!StringUtils::startsWith($haystack, $needle));
        }
    }

    public function testEndsWith(): void
    {
        $haystack = 'blah, bleh bluh';
        foreach (['uh', 'bluh', 'h bluh', ', bleh bluh'] as $needle) {
            $this->assertTrue(StringUtils::endsWith($haystack, $needle));
        }
    }

    public function testEndsWithWrong(): void
    {
        $haystack = 'blah, bleh bluh';
        foreach (['buh', ',bluh', ',blehbluh'] as $needle) {
            $this->assertTrue(!StringUtils::endsWith($haystack, $needle));
        }
    }

    public function testStartsWithAny(): void
    {
        $haystack = 'blah, bleh bluh';
        $needles = ['flah', 'blah'];
        $this->assertTrue(StringUtils::startWithAny($haystack, $needles));
    }

    public function testStartsWithAnyWrong(): void
    {
        $haystack = 'blah, bleh bluh';
        $needles = ['flah', 'fleh'];
        $this->assertTrue(!StringUtils::startWithAny($haystack, $needles));
    }

    public function testEndsWithAny(): void
    {
        $haystack = 'blah, bleh bluh';
        $needles = ['flah', 'bluh'];
        $this->assertTrue(StringUtils::endsWithAny($haystack, $needles));
    }

    public function testEndsWithAnyWrong(): void
    {
        $haystack = 'blah, bleh bluh';
        $needles = ['flah', 'fluh'];
        $this->assertTrue(!StringUtils::endsWithAny($haystack, $needles));
    }

    public function testSafeTruncate(): void
    {
        $string = "blah bleh bluh, bluh bloh\ndlah";

        $this->assertEquals($string, StringUtils::safeTruncate($string, 30));
        $this->assertEquals('blah...', StringUtils::safeTruncate($string, 7));
        $this->assertEquals('blah...', StringUtils::safeTruncate($string, 4));
        $this->assertEquals("blah bleh bluh, bluh bloh...", StringUtils::safeTruncate($string, 28));
        $this->assertEquals('blah bleh!!', StringUtils::safeTruncate($string, 12, '!!'));
        $this->assertEquals('blah bleh bluh, bluh', StringUtils::safeTruncate($string, 20, ''));
    }

    public function testSplitToUniqueParts(): void
    {
        $string = 'blah blah bleh bluh,  bluh bloh';

        $this->assertEquals(
            ['blah', 'bleh', 'bluh,', 'bluh', 'bloh'],
            StringUtils::splitToUniqueParts($string)
        );

        $this->assertEquals(
            ['%blah%', '%bleh%', '%bluh,%', '%bluh%', '%bloh%'],
            StringUtils::splitToUniqueParts($string, fn ($part) => "%{$part}%")
        );

        $this->assertEquals(
            ['%blah%', '%bleh%', '%bluh%', '%bloh%'],
            StringUtils::splitToUniqueParts(
                $string,
                function ($part) {
                    $part = preg_replace('#([^a-zA-Z0-9])#', '', $part); // replace all except chars and numbers
                    return "%{$part}%";
                }
            )
        );
    }
}
