<?php

namespace IZON\Locale;

use IZON\Exceptions\NotImplementedException;
use IZON\Locale\Exceptions\UnableToComposeLocaleTagException;
use IZON\Locale\Exceptions\UnableToParseLocaleTagException;
use Locale as PHPLocale;

/**
 * represents locale
 * can be parsed from localeTag  representation http://php.net/manual/en/class.locale.php from php intl extension
 *
 * lang codes
 * http://www.iana.org/assignments/language-subtag-registry/language-subtag-registry
 * https://docs.oracle.com/cd/E13214_01/wli/docs92/xref/xqisocodes.html
 */
class Locale
{
    /**
     * @var array<string, string>
     */
    protected array $data;

    /**
     * @param array<string, string> $data array with locale data
     */
    protected function __construct(array $data)
    {
        $this->data = $data;
    }

    public function getLanguage(): string
    {
        // language is always present
        return (string)$this->extractValue("language");
    }

    public function getRegion(): ?string
    {
        return $this->extractValue("region");
    }

    public function getScript(): ?string
    {
        return $this->extractValue("script");
    }

    /**
     * @return string serializes locale to locale tag
     * @throws UnableToComposeLocaleTagException if unable to serialize locale to locale tag
     * @throws NotImplementedException
     */
    public function toLocaleTag(): string
    {
        if (extension_loaded('intl')) { // check if intl present
            $result = PHPLocale::composeLocale($this->data);
            if ($result === false) {
                throw new UnableToComposeLocaleTagException(
                    'Unable to serialize to localeTag". For better support install ext-intl.'
                );
            }
            return $result;
        } else {
            throw new NotImplementedException(
                'Unable to serialize to localeTag". For better support install ext-intl.'
            );
        }
    }

    /**
     * @deprecated use self::fromLocaleTag instead
     * @param string $localeTag
     * @return Locale
     * @throws NotImplementedException
     * @throws UnableToParseLocaleTagException
     */
    public static function forLocaleTag(string $localeTag): self
    {
        return self::fromLocaleTag($localeTag);
    }

    /**
     * creates Locale from localeTag string
     * http://php.net/manual/en/class.locale.php
     * @param string $localeTag
     * @return Locale
     * @throws UnableToParseLocaleTagException
     * @throws NotImplementedException
     */
    public static function fromLocaleTag(string $localeTag): self
    {
        if (extension_loaded('intl')) { // check if intl present
            /** @var null|array<string, string> $localeArray */
            $localeArray = PHPLocale::parseLocale($localeTag);

            if ($localeArray === null) {
                throw new UnableToParseLocaleTagException('Unable to parse localeTag "' . $localeTag . '"');
            }
            if (empty($localeArray[PHPLocale::LANG_TAG])) {
                throw new UnableToParseLocaleTagException('Unable to parse localeTag "' . $localeTag . '". Language is missing.');
            }

            $locale = new self($localeArray);
            return $locale;
        } else {
            $localeTag = str_replace("_", "-", $localeTag);
            $localeArray = explode("-", $localeTag);
            if (count($localeArray) == 1 && empty($localeArray[0])) {
                throw new UnableToParseLocaleTagException(
                    "Empty locale provided."
                );
            } elseif (count($localeArray) > 2) {
                throw new UnableToParseLocaleTagException(
                    'Unable to parse localeTag "' . $localeTag . '"."
                    ." Only Language and Region combination is supported in compatibility mode (eg.: cs-CZ, en-US). For better support install ext-intl.'
                );
            } elseif (count($localeArray) > 1) {
                throw new NotImplementedException(
                    'Unable to parse localeTag "' . $localeTag . '"."
                    ." Locale must contain both Language and Region in compatibility mode (eg.: cs-CZ, en-US). For better support install ext-intl.'
                );
            }
            $locale = new Locale([
                'language' => $localeArray[0],
                'region' => $localeArray[1],
            ]);
            return $locale;
        }
    }

    protected function extractValue(string $key): ?string
    {
        return array_key_exists($key, $this->data) ? $this->data[$key] : null;
    }
}
