<?php

namespace IZON\IO;

use IZON\Exceptions\NotImplementedException;
use IZON\Exceptions\UnableToWriteFileException;
use RuntimeException;
use Serializable;
use SplFileObject;

/**
 * temp file that exists only during script run
 */
class TempFile extends AbstractFile implements FileInterface, Serializable
{
    protected string $fileName;

    /**
     * @var resource
     */
    protected $tmpResource;

    /**
     * @var SplFileObject temporary file
     */
    protected SplFileObject $file;

    protected int $fileSize = 0;

    protected ?string $mimeType = null;

    /**
     * @param string $fileName
     */
    public function __construct(string $fileName)
    {
        $this->fileName = $fileName;
        $tmpResource = tmpfile();
        if ($tmpResource === false) {
            throw new RuntimeException("Could not create temporary file.");
        }
        $this->tmpResource = $tmpResource;
        $path = stream_get_meta_data($this->tmpResource)['uri'];
        $this->file = new SplFileObject($path, 'w+');
    }


    /**
     * {@inheritDoc}
     */
    public function getFileContents(): string
    {
        $this->file->rewind();
        $fileContents = $this->file->fread($this->fileSize);
        if ($fileContents === false) {
            throw new RuntimeException("Could not read file contents.");
        }
        return $fileContents;
    }

    /**
     * {@inheritDoc}
     */
    public function getFileExtension(): string
    {
        return pathinfo($this->fileName, PATHINFO_EXTENSION);
    }

    /**
     * {@inheritDoc}
     */
    public function getFileName(): string
    {
        return $this->fileName;
    }

    /**
     * {@inheritDoc}
     */
    public function getFileSize(): int
    {
        return $this->fileSize;
    }

    /**
     * {@inheritDoc}
     */
    public function getFileStream()
    {
        return $this->tmpResource;
    }

    /**
     * {@inheritDoc}
     */
    public function getMimeType(): string
    {
        if (!is_null($this->mimeType)) {
            return $this->mimeType;
        }
        $mime = mime_content_type($this->tmpResource);
        if ($mime === false) {
            throw new NotImplementedException("Fallback when mime is not guessable");
        }
        $this->mimeType = $mime;
        return $mime;
    }

    /**
     * {@inheritDoc}
     */
    public function getTimestamp(): int
    {
        return $this->file->getCTime();
    }

    /**
     * https://www.php.net/manual/en/splfileobject.fwrite.php
     * @param string $data
     * @param null|int $length
     * @return int
     * @throws UnableToWriteFileException
     */
    public function write(string $data, ?int $length = null): int
    {
        /** @var int|false $fileSize */
        $fileSize = false;
        if ($length !== null) {
            $fileSize = $this->file->fwrite($data, $length);
        } else {
            $fileSize = $this->file->fwrite($data);
        }
        /** @phpstan-ignore-next-line */
        if ($fileSize === false) {
            throw new UnableToWriteFileException('Unable to write to tmp file');
        }
        $this->fileSize += $fileSize;
        return $this->fileSize;
    }

    public function getOriginUID(): ?string
    {
        return null;
    }

    public function getFileUID(): string
    {
        return $this->file->getPath();
    }

    /**
     * {@inheritDoc}
     */
    public function serialize()
    {
        throw new NotImplementedException();
    }

    /**
     * {@inheritDoc}
     */
    public function unserialize($serialized)
    {
        throw new NotImplementedException();
    }

    /**
     * @return array<string, mixed>
     * @throws NotImplementedException
     */
    public function __serialize(): array
    {
        throw new NotImplementedException();
    }

    /**
     * @param array<string, mixed> $data
     * @return void
     * @throws NotImplementedException
     */
    public function __unserialize(array $data): void
    {
        throw new NotImplementedException();
    }
}
