<?php

namespace IZON\IO;

use finfo;
use IZON\Exceptions\NotImplementedException;
use IZON\IO\Exceptions\FilesystemAccessException;
use Serializable;
use SplFileInfo;

/**
 * reprezentuje soubor/adresar na disku, je schopno rict jestli se jedna
 * o adresar, soubor potom mimeType, velikost, cas vytvoreni atd.
 *
 * filesystemova cesta se pise s normalnimi lomitky nehlede na os
 * zapisuje se bez koncoveho lomitka
 *
 * $webURI obsahuje cestu pod kterou by mel byt soubor dostupny
 *
 * pouziva: http://php.net/manual/en/class.splfileobject.php
 * http://www.phpro.org/tutorials/Introduction-to-SPL.html#14
 * https://www.inanimatt.com/php-files.html
 *
 */
class File extends AbstractFile implements FileInterface, Serializable
{
    /**
     * @var finfo|null class to get file informations from
     */
    public static ?finfo $fileInfo = null;

    /**
     * @var SplFileInfo kde je ulozeny na fs relativne ke korenu aplikace
     */
    protected SplFileInfo $file;

    /**
     * @var null|string mime type souboru
     */
    protected ?string $mimeType = null;

    /**
     * @param string $path path to file
     */
    public function __construct(string $path)
    {
        $this->file = new SplFileInfo($path);
    }

    /**
     * {@inheritDoc}
     */
    public function getFileContents(): string
    {
        $contents = file_get_contents($this->file->getRealPath());
        if ($contents === false) {
            throw new FilesystemAccessException("Error reading content of file " . $this->file->getRealPath() . ".");
        }
        return $contents;
    }

    /**
     * {@inheritDoc}
     */
    public function getFileExtension(): string
    {
        return $this->file->getExtension();
    }

    /**
     * {@inheritDoc}
     */
    public function getFileName(): string
    {
        return $this->file->getFilename();
    }

    /**
     * {@inheritDoc}
     */
    public function getFileSize(): int
    {
        return $this->file->getSize();
    }

    /**
     * {@inheritDoc}
     */
    public function getFileStream()
    {
        $resource = fopen($this->file->getRealPath(), 'r');
        if ($resource === false) {
            throw new FilesystemAccessException("Error opening file " . $this->file->getRealPath() . " for reading.");
        }
        return $resource;
    }

    /**
     * {@inheritDoc}
     */
    public function getMimeType(): string
    {
        // we have mimeType already created
        if ($this->mimeType !== null) {
            return $this->mimeType;
        }

        if (self::$fileInfo === null) { // init fileinfo, we need only one instance
            self::$fileInfo = new finfo(FILEINFO_MIME_TYPE);
        }

        // initialize mime type of this file
        $mimeType = self::$fileInfo->file($this->file->getRealPath());
        if ($mimeType === false) {
            throw new FilesystemAccessException("Error getting mime type of file " . $this->file->getRealPath() . ".");
        }
        $this->mimeType = $mimeType;

        return $this->mimeType;
    }

    /**
     * {@inheritDoc}
     */
    public function getTimestamp(): int
    {
        return $this->file->getCTime();
    }

    /**
     * @return boolean tests if file exists
     */
    public function exists(): bool
    {
        return file_exists($this->file->getRealPath());
    }

    /**
     * zjisti jestli je
     * @return boolean true pokud je soubor
     */
    public function isDir(): bool
    {
        return $this->file->isDir();
    }

    /**
     * zjisti jestli je soubor
     * @return boolean true if this object is file
     */
    public function isFile(): bool
    {
        return $this->file->isFile();
    }

    /**
     * zjisti jestli je hard nebo soft link na unixech nebo junction na windows?
     */
    public function isLink(): bool
    {
        return $this->file->isLink();
    }

    /**
     * Gets the path without filename
     * @return string
     */
    public function getPath(): string
    {
        return $this->file->getPath();
    }

    /**
     * Gets absolute path to file
     * @return string
     */
    public function getRealPath(): string
    {
        return $this->file->getRealPath();
    }

    public function getOriginUID(): ?string
    {
        return null;
    }

    public function getFileUID(): string
    {
        return $this->file->getPath();
    }

    /**
     * {@inheritDoc}
     */
    public function serialize()
    {
        throw new NotImplementedException();
    }

    /**
     * {@inheritDoc}
     */
    public function unserialize($serialized)
    {
        throw new NotImplementedException();
    }

    /**
     * @return array<string, mixed>
     * @throws NotImplementedException
     */
    public function __serialize(): array
    {
        throw new NotImplementedException();
    }

    /**
     * @param array<string, mixed> $data
     * @return void
     * @throws NotImplementedException
     */
    public function __unserialize(array $data): void
    {
        throw new NotImplementedException();
    }
}
