<?php

namespace IZON\IO;

use Exception;
use finfo;
use Serializable;
use SplFileInfo;

/**
 * reprezentuje soubor/adresar na disku, je schopno rict jestli se jedna
 * o adresar, soubor potom mimeType, velikost, cas vytvoreni atd.
 *
 * filesystemova cesta se pise s normalnimi lomitky nehlede na os
 * zapisuje se bez koncoveho lomitka
 *
 * $webURI obsahuje cestu pod kterou by mel byt soubor dostupny
 *
 * pouziva: http://php.net/manual/en/class.splfileobject.php
 * http://www.phpro.org/tutorials/Introduction-to-SPL.html#14
 * https://www.inanimatt.com/php-files.html
 *
 */
class File extends AbstractFile implements FileInterface {

    /**
     * @var finfo|null class to get file informations from
     */
    static $fileInfo = NULL;

    /**
     * @var SplFileInfo kde je ulozeny na fs relativne ke korenu aplikace
     */
    protected $file;

    /**
     * @var string mime type souboru
     */
    protected $mimeType;
    /**
     * @va string
     */
    protected $path;

    /**
     *
     * @param string $path string, ktery reprezentuje cestu k souboru nebo Adresar do ktereho se ma zapsat novy soubor
     */
    public function __construct($path) {
        if( is_string($path) ) { // is fs path
            $this->file = new SplFileInfo($path);
            $this->path = mb_substr($this->file->getRealPath(), mb_strlen(__BASE_DIR__) + 4); // @phpstan-ignore-line
        } else {
            throw new Exception("Incorrect constructor parameters");
        }
    }


    public function getFileContents(): string {
        return file_get_contents($this->file->getRealPath());
        // DOTO: handle file not readable, not exists etc
    }

    public function getFileExtension(): string {
        return $this->file->getExtension();
    }

    public function getFileName(): string {
        return $this->file->getFilename();
    }

    public function getFileSize(): int {
        return $this->file->getSize();
    }

    public function getFileStream() {
        return fopen($this->file->getRealPath(), 'r');
    }

    public function getMimeType(): string {
        // we have mimeType already created
        if($this->mimeType !== NULL) {
            return $this->mimeType;
        }

        if(self::$fileInfo === NULL) { // init fileinfo, we need only one instance
            self::$fileInfo = new finfo(FILEINFO_MIME_TYPE);
        }

        // initialize mime type of this file
        $this->mimeType = self::$fileInfo->file($this->file->getRealPath());

        return $this->mimeType;
    }

    public function getTimestamp(): int {
        return $this->file->getCTime();
    }

    /**
     * @return boolean tests if file exists
     */
    public function exists(): bool {
        return file_exists($this->file->getRealPath());
    }

    /**
     * zjisti jestli je
     * @return boolean true pokud je soubor
     */
    public function isDir(): bool {
        return $this->file->isDir();
    }

    /**
     * zjisti jestli je soubor
     * @return boolean true if this object is file
     */
    public function isFile(): bool {
        return $this->file->isFile();
    }

    /**
     * zjisti jestli je hard nebo soft link na unixech nebo junction na windows?
     */
    public function isLink(): bool {
        return $this->file->isLink();
    }

    /**
     * Gets the path without filename
     * @return string
     */
    public function getPath(): string {
        return $this->file->getPath();
    }

    /**
     * Gets absolute path to file
     * @return string
     */
    public function getRealPath(): string {
        return $this->file->getRealPath();
    }

    public function getOriginUID(): ?string {
        return null;
    }

    public function getFileUID(): string {
        return $this->path;
    }


    public function __serialize() {
        throw new \IZON\Lang\NotImplementedException();
    }

    public function __unserialize($serialized) {
        throw new \IZON\Lang\NotImplementedException();
    }

}
