<?php


namespace IZON\Utils;


use Exception;
use ReflectionProperty;

/**
 * utility functions for objects
 * @package IZON\Utils
 */
class ObjectUtils {

    /**
     * creates arrow from object, does not convert object properties itself to array
     * includes only nonstatic public and protected properties
     * @param $object
     * @param array $ignoreProperties
     * @return array
     * @throws \ReflectionException
     */
    public static function toArray($object, array $ignoreProperties = []): array {
        if( !is_object($object) ) {
            throw new Exception('$object property is not object');
        }

        $refClass = new \ReflectionClass($object);
        $properties = $refClass->getProperties(ReflectionProperty::IS_PUBLIC | ReflectionProperty::IS_PROTECTED);

        $array = [];
        foreach($properties as $property) {
            if( !in_array($property->getName(), $ignoreProperties) ) { // property is not ignored
                $property->setAccessible(true);
                $array[$property->getName()] = $property->getValue($object);
            }
        }

        return $array;
    }

    /**
     * @param $array
     * @param string $class
     * @param array $ignoreProperties
     */
    public static function fromArray($array, string $class, array $ignoreProperties = []) {
        if( !class_exists($class) ) {
            throw new Exception('$class does not exist');
        }

        $refClass = new \ReflectionClass($class);
        $properties = $refClass->getProperties(ReflectionProperty::IS_PUBLIC | ReflectionProperty::IS_PROTECTED);

        $object = new $class();
        foreach($properties as $property) {
            if( !in_array($property->getName(), $ignoreProperties) ) { // property is not ignored
                $property->setAccessible(true);
                $property->setValue($object, $array[$property->getName()]);
            }
        }

        return $object;
    }

    /**
     * provadi shalow copy
     * kopiruje property z $orig do $dest pres gettery a settery
     * property se kopiruje pokud existuje get* metoda v $orig a set* metoda v $dest
     * @param object $dest to
     * @param object $orig from
     * @param array $ignoreProperties properties to be ignored while copying
     * @throws Exception
     */
    public static function copyProperties($dest, $orig, array $ignoreProperties = []): void {
        if( !is_object($orig) ) {
            throw new Exception('$orig is not object');
        }
        if( !is_object($dest) ) {
            throw new Exception('$dest is not object');
        }

        $methodNames = get_class_methods($orig);
        foreach($methodNames as $methodName) {
            if( StringUtils::startsWith($methodName, "get") ) { // existuje get metoda
                $propertyName = mb_substr($methodName, 3);
                $propertyName = mb_strtolower(mb_substr($propertyName, 0, 1)) . mb_substr($propertyName, 1);

                $setMethodName = str_replace("get", "set", $methodName);

                if( !in_array($propertyName, $ignoreProperties)
                    && method_exists($dest, $setMethodName) ) { // existuje set metoda
                    $val = $orig->$methodName();
                    $dest->$setMethodName($val);
                }
            }
        }
    }

    function createGetterMethodName(string $propertyName): string {
        $methodName = "get". ucfirst($propertyName);
        return $methodName;
    }

    function createIsMethodName(string $propertyName): string {
        $methodName = "is". ucfirst($propertyName);
        return $methodName;
    }

    function createSetterMethodName(string $propertyName): string {
        $methodName = "set". ucfirst($propertyName);
        return $methodName;
    }
}
