<?php

namespace IZON\Utils;

use DateInterval;

use DateTime;
use DateTimeZone;
use Exception;

/**
 * trida reprezentujici datum a cas
 * umoznuje vytvoreni z timestamp a z hodin, minut., sekund, dnu, tydnu, mesicu, let
 */
class Date
{
    /**
     * @var DateTime
     */
    private $dateTime;

    /**
     * pokud neni zadana timestamp vytvori se datum s aktualnim casem
     * @param integer $timestamp timestamp reprezetujici vytvarene datum
     * @param mixed $timezone Description
     */
    public function __construct($timestamp = null, $timezone = null)
    {
        if ($timestamp === null) {
            if ($timezone === null) {
                $this->dateTime = new DateTime("now");
            } else {
                $this->dateTime = new DateTime("now", $timezone);
            }
        } elseif ($timestamp instanceof DateTime) {
            $this->dateTime = $timestamp;
        } elseif (is_long($timestamp)) { // nastavuje timestamp
            $this->dateTime = new DateTime();
            $this->dateTime->setTimestamp($timestamp);
        } else {
            throw new Exception("Unsupported paramters.");
        }
    }

    /**
     * equivalet to http://php.net/manual/en/datetime.createfromformat.php
     * @param string $format
     * @param string $time
     * @param DateTimeZone|null $timezone
     * @param boolean $strict if use strict format
     * @return boolean|Date returns Date if ok or false if any error occured
     */
    public static function createFromFormat($format, $time, ?DateTimeZone $timezone = null, $strict = true)
    {
        if ($strict) {
            $format = '!' . $format;
        }
        if ($timezone === null) {
            $datetime = DateTime::createFromFormat($format, $time);
        } else {
            $datetime = DateTime::createFromFormat($format, $time, $timezone);
        }
        if ($datetime == false) {
            return false;
        } else {
            return new Date($datetime);
        }
    }

    /**
     * @param string $format format format ve stylu php funkce date http://cz.php.net/manual/en/function.date.php a czdate
     * @return string formatovane datum podle zadaneho parametru
     */
    public function format($format)
    {
        return $this->dateTime->format($format);
    }

    /**
     * @return integer vraci timestamp odpovidajici tomuto datu
     */
    public function getTimestamp()
    {
        return $this->dateTime->getTimestamp();
    }

    /**
     * vraci true, pokud je to datum pred zadanym datem
     * v pripade za zadana hodnota neni datum vyvolava exception
     * @param Date $date datum u ktereho se ma zjistit jestli je mensi
     * @return boolean
     */
    public function before(Date $date)
    {
        return $this->getTimestamp() < $date->getTimestamp();
    }

    /**
     * vraci true, pokud je to datum vetsi nez zadane
     * v pripade za zadana hodnota neni datum vyvolava exception
     * @param Date $date datum u ktereho se ma zjistit jestli je vetsi
     * @return boolean
     */
    public function after(Date $date)
    {
        return $this->getTimestamp() > $date->getTimestamp();
    }

    /**
     * vraci true, pokud jsou data stejna
     * v pripade za zadana hodnota neni datum vyvolava exception
     * @param Date $date datum, ktere se ma porovnat
     * @return boolean
     */
    public function equals(Date $date)
    {
        return $this->getTimestamp() == $date->getTimestamp();
    }

    /**
     * 0 - equalDates,
     * $thisDate > $dataParams = more than 0,
     * $thisDate < $dataParams = less than 0,
     * the value 0 if the argument Date is equal to this Date; a value less than 0 if this Date is before the Date argument; and a value greater than 0 if this Date is after the Date argument.
     */
    public function compareTo($date)
    {
        if (!is_a($date, self::class)) {
            throw new Exception("$date musi byt instance date.");
        }
        return $this->getTimestamp() - $date->getTimestamp();
    }

    /**
     * @return DateTime
     */
    public function getDateTime()
    {
        return $this->dateTime;
    }

    /**
     * Add interval to this object
     * @param int $seconds num to add
     * @param int $minutes num to add
     * @param int $hours num to add
     * @param int $days num to add
     * @param int $months num to add
     * @param int $years num to add
     *
     * $seconds should be required as parameter
     */
    public function addTime($seconds = 0, $minutes = 0, $hours = 0, $days = 0, $months = 0, $years = 0)
    {
        $interval = self::makeInterval($seconds, $minutes, $hours, $days, $months, $years);
        $this->dateTime->add($interval);
    }

    /**
     * Sub interval from this object
     * @param int $seconds num to sub
     * @param int $minutes num to sub
     * @param int $hours num to sub
     * @param int $days num to sub
     * @param int $months num to sub
     * @param int $years num to sub
     *
     * $seconds should be required as parameter
     */
    public function subTime($seconds = 0, $minutes = 0, $hours = 0, $days = 0, $months = 0, $years = 0)
    {
        $interval = self::makeInterval($seconds, $minutes, $hours, $days, $months, $years);
        $this->dateTime->sub($interval);
    }

    /**
     * Make Date interval from number of time arguments
     * @param int $seconds count of seconds in interval
     * @param int $minutes count of minutes in interval
     * @param int $hours count of hours in interval
     * @param int $days count of days in interval
     * @param int $months count of months in interval
     * @param int $years count of years in interval
     * @return \DateInterval
     *
     * makeInterval should be brobbaby protected method
     * $seconds should be required as parameter
     */
    public static function makeInterval($seconds = 0, $minutes = 0, $hours = 0, $days = 0, $months = 0, $years = 0)
    {
        foreach (func_get_args() as $key => $arg) {
            if (!is_numeric($arg)) {
                //number of argument start on 0
                throw new Exception('Interval of argument number: ' . $key . ' is not integer');
            }
            if ($arg < 0) {
                //number of argument start on 0
                throw new Exception('Interval of argument number: ' . $key . ' can not be negative');
            }
        }
        $intervalString = "";

        $intervalString .= "P";
        $intervalString .= $years . "Y";
        $intervalString .= $months . "M";
        $intervalString .= $days . "D";

        $intervalString .= "T";
        $intervalString .= $hours . "H";
        $intervalString .= $minutes . "M";
        $intervalString .= $seconds . "S";

        $interval = new DateInterval($intervalString);
        return $interval;
    }
    /**
     * return true if day is between dates
     * @param Date $date1
     * @param Date $date2
     * @return boolean
     */
    public function between(Date $date1, Date $date2)
    {
        return ($this->after($date1) && $this->before($date2)) || ($this->after($date2) && $this->before($date1));
    }

    public function __clone()
    {
        $this->dateTime = clone $this->dateTime;
    }
}
