<?php

namespace IZON\IO;

use Exception;
use finfo;
use IZON\Utils\Date;
use Serializable;
use SplFileInfo;
use SplFileObject;

/**
 * reprezentuje soubor/adresar na disku, je schopno rict jestli se jedna
 * o adresar, soubor potom mimeType, velikost, cas vytvoreni atd.
 *
 * filesystemova cesta se pise s normalnimi lomitky nehlede na os
 * zapisuje se bez koncoveho lomitka
 *
 * $webURI obsahuje cestu pod kterou by mel byt soubor dostupny
 *
 * pouziva: http://php.net/manual/en/class.splfileobject.php
 * http://www.phpro.org/tutorials/Introduction-to-SPL.html#14
 * https://www.inanimatt.com/php-files.html
 *
 */
class File implements Serializable
{
    /**
     * @var finfo|null class to get file informations from
     */
    public static $fileInfo = null;
    /**
     * @var string mime type souboru
     */
    protected $mimeType;
    /**
     * @var SplFileInfo kde je ulozeny na fs relativne ke korenu aplikace
     */
    protected $file;

    /**
     * relativni cesta k souboru
     * @var string
     */
    //    protected $relativePath;
    /**
     * @var string|NULL pod jakou cestou je dostupne na webu od korene webu, pokud je null tak soubor neni pristupny z webu
     *
     */
    protected $webURI;

    /**
     *
     * @param string $path string, ktery reprezentuje cestu k souboru nebo Adresar do ktereho se ma zapsat novy soubor
     * @param string $webURI pod jakou cestou ma byt dostupne na webu
     */
    public function __construct($path, $webURI = null)
    {
        if (is_string($path)) { // is fs path
            $this->file = new SplFileInfo($path);
        } else {
            throw new Exception("Incorrect constructor parameters");
        }

        if ($webURI != null) {
            $this->webURI = $webURI;
        } else {
            // pokud neni zadano web uri tak se bere relativne k workdir
            $this->webURI = str_replace(\IZON\File\normalizeFileName(getcwd()), "", \IZON\File\normalizeFileName($path));
        }
    }

    /**
     * @return boolean tests if file exists
     */
    public function exists()
    {
        return file_exists($this->file->getRealPath());
    }

    /**
     * zjisti jestli je
     * @return boolean true pokud je soubor
     */
    public function isDir()
    {
        return $this->file->isDir();
    }

    /**
     * zjisti jestli je soubor
     * @return boolean true if this object is file
     */
    public function isFile()
    {
        return $this->file->isFile();
    }

    /**
     * zjisti jestli je hard nebo soft link na unixech nebo junction na windows?
     */
    public function isLink()
    {
        return $this->file->isLink();
    }

    /**
     * vrati datum posledni modifikace souboru
     * @return Date datum posledni modifikace pokud je dostupna
     */
    public function lastModifiedDate()
    {
        return new Date($this->file->getATime());
    }

    /**
     * vrati datum posledni modifikace souboru
     * @return Date datum posledni modifikace pokud je dostupna
     */
    public function createdDate()
    {
        return new Date($this->file->getCTime());
    }

    /**
     * @deprecated use getMimeType
     * @return string vraci mime-type
     */
    public function getMineType()
    {
        return $this->getMimeType();
    }

    /**
     * @return string vraci mime-type
     */
    public function getMimeType()
    {
        // we have mimeType already created
        if ($this->mimeType !== null) {
            return $this->mimeType;
        }

        if (self::$fileInfo === null) { // init fileinfo, we need only one instance
            self::$fileInfo = new finfo(FILEINFO_MIME_TYPE);
        }
        if ($this->mimeType === null) {
            // initialize mime type of this file
            $this->mimeType = self::$fileInfo->file($this->file->getRealPath());
        }
        return $this->mimeType;
    }

    /**
     * vrati absolutni path, kde se soubor nachazi na disku
     */
    public function getFsPath()
    {
        return $this->file->getRealPath();
    }

    /**
     *
     * @deprecated should not by used, pass through gallery
     */
    public function getWebURI()
    {
        return $this->webURI;
    }

    /**
     *
     * @return string vraci v jakem adresari se nachazi
     */
    public function getDir()
    {
        return $this->file->getPath();
    }

    /**
     * cisty nazev souboru bez adresare kde se nachazi
     * @return string
     */
    public function getFileName()
    {
        return $this->file->getFilename();
    }

    public function getExtension()
    {
        return $this->file->getExtension();
    }

    /**
     * @return string|null returns
     */
    public function getFileContent()
    {
        if ($this->file->isReadable()) {
            return file_get_contents($this->getFsPath());
        }
        return null;
    }

    /**
     * return file size in bytes
     * @return int
     */
    public function getFileSize()
    {
        return $this->file->getSize();
    }

    public function getFriendlyFileSize()
    {
        $decimal = 2;
        $ret['raw'] = $this->file->getSize();

        if (is_numeric($ret['raw'])) {
            $position = 0;
            $units = ["B", " KB", " MB", " GB", " TB"];
            while ($ret['raw'] >= 1024 && ($ret['raw'] / 1024) >= 1) {
                $ret['raw'] /= 1024;
                $position++;
            }
            $ret['friendly'] = round($ret['raw'], $decimal) . $units[$position];
        } else {
            $ret['friendly'] = "0 Bytes";
        }

        return $ret['friendly'];
    }

    /**
     * Return file extension example 'zip' , 'jpg'
     * @return string
     */
    public function getFileExtension()
    {
        return $this->file->getExtension();
    }

    /**
     * deletes file from file system
     */
    public function delete()
    {
        $path = $this->file->getRealPath();
        unset($this->file);
        unlink($path);
    }

    /**
     * Zapise do souboru
     * @param string $string
     * @return integer pocet zapsanych znaku
     */
    public function write($string)
    {
        if ($this->exists() && $this->isFile()) {
            $tempFile = new SplFileObject($this->getFsPath(), "w");
            return $tempFile->fwrite($string);
        }
        return -1;
    }

    /**
     * TODO: it shoud return new instance of File not absolute file path
     * creates new instande of file in $path
     * @param string $path
     * @return string return
     */
    public static function createFile($path)
    {
        $obj = new SplFileObject($path, "w");
        return $obj->getRealPath();
    }

    /**
     * returns "safe" file name ie. removes unsafe characters that can make problems on windows or linux fs
     * also conversts all characters to lowercase
     * @param string $name
     * @return string
     */
    public static function getSafeFileName($name)
    {
        $fileName = strtolower(pathinfo($name, PATHINFO_FILENAME));
        $fileExtension = pathinfo($name, PATHINFO_EXTENSION);

        $frontFrom = [' ', '&aacute;', '&Aacute;', '&auml;', '&Auml;', '&eacute;', '&Eacute;', '&euml;', '&Euml;', '&iacute;', '&Iacute;', '&iuml;', '&Iuml;', '&oacute;', '&Oacute;', '&ouml;', '&Ouml;', '&scaron;', '&Scaron;', '&uacute;', '&Uacute;', '&uuml;', '&Uuml;', '&yacute;', '&Yacute;', '&yuml;', '&Yuml;'];
        $frontTo = ['-', 'a', 'a', 'a', 'a', 'e', 'e', 'e', 'e', 'i', 'i', 'i', 'i', 'o', 'o', 'o', 'o', 's', 's', 'u', 'u', 'u', 'u', 'y', 'y', 'y', 'y'];

        static $convertTable = [
            'á' => 'a', 'Á' => 'A', 'ä' => 'a', 'Ä' => 'A', 'č' => 'c',
            'Č' => 'C', 'ď' => 'd', 'Ď' => 'D', 'é' => 'e', 'É' => 'E',
            'ě' => 'e', 'Ě' => 'E', 'ë' => 'e', 'Ë' => 'E', 'í' => 'i',
            'Í' => 'I', 'ï' => 'i', 'Ï' => 'I', 'ľ' => 'l', 'Ľ' => 'L',
            'ĺ' => 'l', 'Ĺ' => 'L', 'ň' => 'n', 'Ň' => 'N', 'ń' => 'n',
            'Ń' => 'N', 'ó' => 'o', 'Ó' => 'O', 'ö' => 'o', 'Ö' => 'O',
            'ř' => 'r', 'Ř' => 'R', 'ŕ' => 'r', 'Ŕ' => 'R', 'š' => 's',
            'Š' => 'S', 'ś' => 's', 'Ś' => 'S', 'ß' => 'S',
            'ť' => 't', 'Ť' => 'T',
            'ú' => 'u', 'Ú' => 'U', 'ů' => 'u', 'Ů' => 'U', 'ü' => 'u',
            'Ü' => 'U', 'ý' => 'y', 'Ý' => 'Y', 'ÿ' => 'y', 'Ÿ' => 'Y',
            'ž' => 'z', 'Ž' => 'Z', 'ź' => 'z', 'Ź' => 'Z',
        ];
        $newFileName = str_replace($frontFrom, $frontTo, $fileName);
        $newFileName = mb_strtolower(strtr($newFileName, $convertTable), 'UTF-8');
        $newFileName = preg_replace('/[^a-zA-Z0-9\-_]+/u', '-', $newFileName);
        $newFileName = trim($newFileName);

        if (trim($newFileName) == '') {
            // pokud je celej nazev souboru spatne, tak se mu da nejaky defaultni nazev ve tvaru "default_[rand(0, 1000000)].[puvodni_pripona]"
            $newFileName = 'default_' . rand(0, 1000000);
        }

        $parts = array_filter([$newFileName, $fileExtension]);
        return strtolower(implode('.', $parts));
    }

    /**
     * implements Serializable interface serialize
     * @return string serialized object
     */
    public function serialize()
    {
        $serializeArray = $this->__serialize();
        return serialize($serializeArray);
    }

    /**
     * implements Serializable interface serialize
     * @param string $serialized
     */
    public function unserialize($serialized)
    {
        $unserializedArray = unserialize($serialized);
        $this->__unserialize($unserializedArray);
    }

    public function __serialize(): array
    {
        $serializeArray = [];
        $serializeArray["filePath"] = $this->file->getRealPath();
        $serializeArray["mimeType"] = $this->mimeType;
        return $serializeArray;
    }

    public function __unserialize(array $data): void
    {
        $this->file = new SplFileInfo($data["filePath"]);
        $this->mimeType = $data["mimeType"];
    }
}
