<?php
  namespace IZON\Utils;
  
  // TODO: is handled by BaseAdminService|Controller probabbly can be removed in future
  class FileUloader {

    public static $VERSION = '1.0';
  
   /**
     * promenna obsahuje zakladni adresar, kam se budou soubory uploadovat
     * 
     * @var string
     */
    protected $baseDir;
    
    /**
     * promenna obsahuje nastaveni uploadu souboru
     * 
     * @var array
     */
    protected $params;
    
    /**
     * zakazane pripony souboru
     * 
     * @var array          
     */          
    protected static $BANNED_EXTENSIONS = array('php','php3','php4','php5','inc');
    /**
     * defaultni adresar pro nahrani souboru
     *      
     * @var array
     */           
    protected static $DEFAULT_BASEDIR = '/upload/temp/';
    
    /**
     * Konstruktor
     * 
     * pokud bude $params zadano:<br>
     * $params['maxFileSize']         - maximalni velikost uploadovaneho suboru v bajtech<br>
     * $params['checkFileExtension']  - [true | false] - pokud se maji kontrolovat podezdrele pripony souboru. Defaultne zapnute<br>
     * $params['addTimeStamp']        - [true | false] - pokud se ma k nazvu souboru pripojit jednoznacne cislo (unix time stamp). Defaultne vypnuto.<br>
     * $params['newName']             - {[cislo][hodnota]} jmena, pod kterym budou soubory ulozeny (jmeno je prohnano pres validator a "skodlive" znaky budou nahrazeny "-").
     *                                Dale je potreba, aby pole zacinalo od 0, tj. aby "cislo" prvniho prvku pole byla 0.<br>
     * $params['acceptedExtensions'] - pole s priponami akceptovanych souboru, napr.: array('jpg', 'png')<br><br>
     * Nebude-li zadan newName, pouzije se puvodni nazev souboru.<br>
     * Nebude-li zadan $baseDir, pouzije se defaultne self::$DEFAULT_BASEDIR.      
     * 
     * @param string $baseDir - zakladni adresar 
     * @param array $params - nepovinna polozka, umoznujici nektera dalsi nastaveni.
     * @access public                              
     */              
    public function __construct($baseDir, $params = array()) {
      $this->setBasedir($baseDir);
      $this->params = $params;
    }
    
    /**
     * funkce nastavuje zakladni adresar pro upload souboru
     * 
     * @param string $basedir   
     * @return no return value            
     * @access public
     */              
    protected function setBasedir($basedir) {
      $basedir = trim($basedir);
      if($basedir != '') {
        if(substr($basedir, strlen($basedir)-1, strlen($basedir)) != '/') {
          $basedir = $basedir.'/';
        }
        $this->baseDir = $basedir;
      } else {
        // pokud nebude zadan zadny z adresaru, zvoli defaultni
        $this->baseDir = self::$DEFAULT_BASEDIR;
      }
      // Pokud basedir nebude existovat, vytvori se. Pokud vytvoreni selze, nastavi se basedir na defaultni
      if(!is_dir($this->baseDir)) {
        if(mkdir($this->baseDir)) {
          chmod($this->baseDir, 0777);
        } else {
          $this->baseDir = self::$DEFAULT_BASEDIR;
        }
      }
    }
    
    /**
     * funkce vraci zakladni adresar pro upload souboru
     * 
     * @return string   
     * @access public
     */  
    public function getBasedir() {
      return $this->baseDir;
    } 
    
    /**
     * Funkce na upload souboru.
     * 
     * $files - uploadovane soubory        
     * vraci pole $ret[uploaded][_counter_]['origName'] - puvodni jmeno souboru<br>
     *            $ret[uploaded][_counter_]['newName'] - nove jmeno souboru (je-li zadano)<br>
     *            $ret[uploaded][_counter_]['inputName']<br>
     * V pripade selhani uploadu souboru: <br>
     *            $ret[failed][_counter][file] - puvodni nazev souboru
     *            $ret[failed][_counter][error] - chybova hlaska
     * 
     * @param array $files
     * @param boolean $multiply - jestli se jedna o nahravani jednoho nebo vice souboru, defaultne false
     * @access public
     * @return array                   
     *       
     */       
     public function run($files, $multiply = false) {
       if($multiply) {
         $files = $this->removeEmptyFields($files);
       }
       $ret['uploaded'] = null;
       $ret['failed']   = null;
       if(is_array($files)) {
         $counter = 0;
         $uploaded_file = false;
         foreach ($files as $key => $file) {
           if(trim($file['name']) == '') {
             $uploaded_file = true;
             continue;
           }
           try {
              $this->checkFileExtension($file['tmp_name']);
              $this->checkFileSize($file['size']);
              
              $fileName = $this->generateFileName($file['name'], $this->params['newName'][$counter]);
              #
              $currentFile['origName']   = $file['name'];
              $currentFile['newName']    = $fileName;
              $currentFile['inputName']  = $key;
              #
              
              if(move_uploaded_file($file['tmp_name'],$this->baseDir.$fileName)) {
                chmod($this->baseDir.$fileName, 0777);
                #
                $ret['uploaded'][$counter]  = $currentFile;
                #
                $uploaded_file = true;
              } else {
                switch($value['error']) {
                  case UPLOAD_ERR_INI_SIZE:      
                  case UPLOAD_ERR_FORM_SIZE:     throw new \Exception($file['name'], 2);  break;
                  default: throw new \Exception($file['name'], 3); 
                }
              }
              
              // pokud bude zadano pridavani time stamp k nazvu souboru, tak si chvilku pockame, abys nedoslo k tomu,
              // ze by time stamp byl stejny
              
              
           } catch (\Exception $exc) {
             $ret['failed'][$counter]['file']  = $file['name'];
             switch ($exc->getCode()) {
               case '1': $ret['failed'][$counter]['error'] = 'BANNED_FILE_EXTENSION'; break;
               case '2': $ret['failed'][$counter]['error'] = 'FILE_SIZE_IS_OUT_OF_SCOPE'; break;
               case '3': $ret['failed'][$counter]['error'] = 'FILE_UPLOAD_FAIL'; break;
             }
           }
           
           $counter++;
           if($params['addTimeStamp']) {
             usleep(100);
           }
         }
       }
       $ret['state'] = $uploaded_file; 
       return $ret;
     }
     /**
      * Vygeneruje nazev souboru, pod kterym bude ulozen
      * 
      * @param string $origFileName - nazev puvodniho souboru
      * @param string $newName - nazev noveho souboru
      * @return string
      */
     public function generateFileName($origFileName, $newName = null) {
       $extension = pathinfo($origFileName, PATHINFO_EXTENSION);
       $_file     = $origFileName;
       if(!is_null($newName)) {
         $_file = $newName.'.'.$extension;
       }
       $fileName = $this->getSafeFileName($_file);
       if($this->params['addTimeStamp']) {
         $fileName = str_replace(array('.', ','), '', microtime(true)).'-'.$fileName;
       } 
       return $fileName;
     }
    
    /**
     * funkce vraci "bezpecny" nazev souboru a prevede jej na mala pismena
     * 
     * @param string $name
     * @return string 
     * @access protected                    
     */              
    public static function getSafeFileName($name) {
      $fileName       = strtolower(pathinfo($name, PATHINFO_FILENAME));
      $fileExtension  = pathinfo($name, PATHINFO_EXTENSION);

      $frontFrom = array (' ', '&aacute;', '&Aacute;', '&auml;', '&Auml;', '&eacute;', '&Eacute;', '&euml;', '&Euml;', '&iacute;', '&Iacute;', '&iuml;', '&Iuml;', '&oacute;', '&Oacute;', '&ouml;', '&Ouml;', '&scaron;', '&Scaron;', '&uacute;', '&Uacute;', '&uuml;', '&Uuml;', '&yacute;', '&Yacute;', '&yuml;', '&Yuml;');
      $frontTo = array ('-', 'a', 'a', 'a', 'a', 'e', 'e', 'e', 'e', 'i', 'i', 'i', 'i', 'o', 'o', 'o', 'o', 's', 's', 'u', 'u', 'u', 'u', 'y', 'y', 'y', 'y');

	    static $convertTable = array (
	        'á' => 'a', 'Á' => 'A', 'ä' => 'a', 'Ä' => 'A', 'č' => 'c',
	        'Č' => 'C', 'ď' => 'd', 'Ď' => 'D', 'é' => 'e', 'É' => 'E',
	        'ě' => 'e', 'Ě' => 'E', 'ë' => 'e', 'Ë' => 'E', 'í' => 'i',
	        'Í' => 'I', 'ï' => 'i', 'Ï' => 'I', 'ľ' => 'l', 'Ľ' => 'L',
	        'ĺ' => 'l', 'Ĺ' => 'L', 'ň' => 'n', 'Ň' => 'N', 'ń' => 'n',
	        'Ń' => 'N', 'ó' => 'o', 'Ó' => 'O', 'ö' => 'o', 'Ö' => 'O',
	        'ř' => 'r', 'Ř' => 'R', 'ŕ' => 'r', 'Ŕ' => 'R', 'š' => 's',
	        'Š' => 'S', 'ś' => 's', 'Ś' => 'S', 'ß' => 'S',
          'ť' => 't', 'Ť' => 'T',
	        'ú' => 'u', 'Ú' => 'U', 'ů' => 'u', 'Ů' => 'U', 'ü' => 'u',
	        'Ü' => 'U', 'ý' => 'y', 'Ý' => 'Y', 'ÿ' => 'y', 'Ÿ' => 'Y',
	        'ž' => 'z', 'Ž' => 'Z', 'ź' => 'z', 'Ź' => 'Z',
	    );
	    $string = str_replace($frontFrom, $frontTo, $fileName);
    	$string = mb_strtolower(strtr($string, $convertTable), 'UTF-8');
    	$string = preg_replace('/[^a-zA-Z0-9]+/u', '-', $string);
    	$string = trim($string);

      if(trim($string) == '') {
        // pokud je celej nazev souboru spatne, tak se mu da nejaky defaultni nazev ve tvaru "default_[rand(0, 1000000)].[puvodni_pripona]"
        $string = 'default_'.rand(0, 1000000);
      }
      return strtolower($string.'.'.$fileExtension);
    }
    
    /**
     * smaze vsecka prazdna pole. 
     * Dobre predefinovat v potomkovi, pokud se uploadovane soubory predavaji v poli. 
     * Proste si pripadne parametry preskladame tak, jak je ocekava metoda move();
     * 
     * @param array $data
     * @return mixed
     * @access protected               
     */              
    protected function removeEmptyFields($data) {
      $ret = '';
      if(is_array($data['name'])) {
        foreach ($data['name'] as $key=>$value) {
        	if(trim($value) != '') {
        	  $ret[$key]['name'] = $value;
        	  $ret[$key]['type'] = $data['type'][$key];
        	  $ret[$key]['tmp_name'] = $data['tmp_name'][$key];;
        	  $ret[$key]['error'] = $data['error'][$key];;
        	  $ret[$key]['size'] = $data['size'][$key];;
          }
        }
      } else {
        $ret = '';
      }
      return $ret;
    }
  
    /**
     * Zjisti, jestli upladovany soubor neni moc veliky
     * 
     * @param type $file_size
     * @throws \Exception
     */
     protected function checkFileSize($file_size) {
       if(isset($this->params['maxFileSize'])) {
         $max_upload_size = round($this->params['maxFileSize'], 0);
       } else {
         $max_upload_size = min($this->let_to_num(ini_get('post_max_size')), $this->let_to_num(ini_get('upload_max_filesize'), $this->let_to_num(ini_get('memory_limit'))));
       }
       if($file_size > $max_upload_size) {
         throw new \Exception('FILE_SIZE_IS_OUT_OF_SCOPE', 2);
       } 
     }
     /**
     * This function transforms the php.ini notation for numbers (like '2M') to an integer (2*1024*1024 in this case)
     *
     * @param int $v
     * @return int
     * @access protected         
     */   
     protected function let_to_num($v) { //
       $l = substr($v, -1);
       $ret = substr($v, 0, -1);
       switch(strtoupper($l)){
       case 'P':
          $ret *= 1024;
       case 'T':
          $ret *= 1024;
       case 'G':
          $ret *= 1024;
       case 'M':
          $ret *= 1024;
       case 'K':
          $ret *= 1024;
          break;
       }
       return $ret;
    }
    
    /**
     * Kontroluje, jestli uploadovany soubor nema zakazanou priponu
     * 
     * @param type $file
     * @throws \Exception
     */
    protected function checkFileExtension($file) {
      $finfo = finfo_open(FILEINFO_MIME_TYPE); // return mime type ala mimetype extension
      $mineType = explode('/',finfo_file($finfo, $file));
      finfo_close($finfo);
      $fileExtension = $mineType[1];
      if($this->params['checkFileExtension'] && in_array($fileExtension, self::$BANNED_EXTENSIONS)) {
        throw new \Exception('BANNED_FILE_EXTENSION', 1);
      } elseif(!empty($this->params['acceptedExtensions']) && !in_array(strtolower($fileExtension), $this->params['acceptedExtensions'])) {
        throw new \Exception('BANNED_FILE_EXTENSION', 1);
      }
    }
  }
