<?php

namespace IZON\Users\Web\Controllers;

use Exception;
use IZON\MVC\Context\Context;
use IZON\MVC\Controller;
use IZON\MVC\HttpRequest;
use IZON\MVC\Json\JsonResponse;
use IZON\MVC\ModelAndView;
use IZON\Users\Presentation\Domain\UserInfo;
use IZON\Users\Services\LoggedWebUserSessionServiceInterface;
use IZON\Users\Web\Forms\UserWebFormFactory;

/**
 *
 */
class LoginController implements Controller
{
    protected LoggedWebUserSessionServiceInterface $sessionService;

    /**
     * controoler na ktery se ma po prihlaseni presmerovat
     * @var string
     */
    protected $redirectControllerIdentifier = null;

    /**
     * jake view se ma pouzit pro zobrazeni formulare
     * @var string
     */
    protected $loginFormViewName = "users/unlogged/login";


    public function __construct(LoggedWebUserSessionServiceInterface $sessionService)
    {
        $this->sessionService = $sessionService;
    }

    public function execute(HttpRequest $request, Context $context)
    {
        $loginForm = $this->createLoginForm($request);
        $loginForm->setValues($request->getParameters());

        // neni vyplneny zadny login
        if ($loginForm->getField("login")->parse() == '') {
            $modelAndView = new ModelAndView($this->loginFormViewName, ["loginForm" => $loginForm]);
            return $modelAndView;
        }

        $login = $loginForm->getField("login")->parse();
        $password = $loginForm->getField("password")->parse();

        // pokusit se prihlasit
        try {
            $this->sessionService->loginUser($login, $password);
            $modelAndView = new ModelAndView($this->redirectControllerIdentifier, [], true);
            return $modelAndView;
        } catch (Exception $e) {
            $errorMessage = $this->translateExceptionToFormErrorMessage($e, $login, $context);
            $loginForm->addFormError($errorMessage);
            $modelAndView = new ModelAndView($this->loginFormViewName, ["loginForm" => $loginForm, "loginError" => true]);
        }

        return $modelAndView;
    }

    protected function createLoginForm(HttpRequest $request)
    {
        return UserWebFormFactory::createLoginForm();
    }

    protected function translateExceptionToFormErrorMessage(Exception $exception, $login, Context $context)
    {
        $errorMessage = "";
        switch ($exception->getCode()) {
            case 100:
                $errorMessage = sprintf(_("Uživatel %s neexistuje."), $login);
                break;
            case 200:
                $errorMessage = sprintf(_("Pro uživatele %s nesouhlasí heslo."), $login);
                break;
            case 300:
                $errorMessage = sprintf(_("Uživatel %s není povolený."), $login);
                break;
            case 400:
                $errorMessage = sprintf(_("Účet uživatele %s není kompletní."), $login);
                break;
            default:
                $errorMessage = $exception->getMessage();
                break;
        }
        return $errorMessage;
    }

    public function ajaxLogin(HttpRequest $request, Context $context)
    {
        $loginForm = $this->createLoginForm($request);
        $loginForm->setValues($request->getParameters());

        $login = $loginForm->getField("login")->parse();
        $password = $loginForm->getField("password")->parse();

        $modelAndView = new ModelAndView("simple/json-respose");

        try {
            $this->sessionService->loginUser($login, $password);
            $user = $this->sessionService->getLoggedUser();
        } catch (Exception $exception) {
            $errorMessage = $this->translateExceptionToFormErrorMessage($exception, $login, $context);
            $result = JsonResponse::createErrorResponse($errorMessage);
            $modelAndView->putParameter("json", $result);
            return $modelAndView;
        }

        $result = JsonResponse::createResponse($this->serializeUserInfo($user));
        $modelAndView->putParameter("json", $result);

        return $modelAndView;
    }

    protected function serializeUserInfo(UserInfo $user)
    {
        $usetArray = [
            "name" => $user->getName(),
            "email" => $user->getEmail()
        ];
        return $usetArray;
    }

    public function setRedirectControllerIdentifier($redirectControllerIdentifier)
    {
        $this->redirectControllerIdentifier = $redirectControllerIdentifier;
    }

    public function setLoginFormViewName($loginFormViewName)
    {
        $this->loginFormViewName = $loginFormViewName;
    }
}
