<?php

namespace IZON\Users\Services;

use Exception;
use IZON\Mailer\Mailer;
use IZON\Users\Dao\RoleDao;
use IZON\Users\Dao\UserDao;
use IZON\Users\Dao\UserResetPasswordRequestDao;
use IZON\Users\Dao\UserRoleDao;
use IZON\Users\Domain\User;
use IZON\Users\Domain\UserResetPasswordRequest;
use IZON\Users\Presentation\Domain\UserInfo;
use IZON\Utils\Date;

/**
 * Servis provadejici ostavu session ziskavane z db AdminUser
 */
abstract class AbstractLoggedWebUserSessionService implements LoggedWebUserSessionServiceInterface
{
    /**
     * @var Mailer mailer pro odesilani emailu
     */
    protected Mailer $mailer;

    protected UserDao $userDao;

    protected RoleDao $roleDao;

    protected UserRoleDao $userRoleDao;

    protected UserResetPasswordRequestDao $userResetPasswordRequestDao;


    /**
     *
     * @var string name of session variable where is stored logged user
     */
    protected string $userSessionName = "loggedUser";

    /**
     * @var int minimal length of password
     */
    protected int $minPasswordLength = 5;


    /// generovany konstruktor
    public function __construct(
        Mailer $mailer,
        UserDao $userDao,
        RoleDao $roleDao,
        UserRoleDao $userRoleDao,
        UserResetPasswordRequestDao $userResetPasswordRequestDao
    ) {
        $this->mailer = $mailer;
        $this->userDao = $userDao;
        $this->roleDao = $roleDao;
        $this->userRoleDao = $userRoleDao;
        $this->userResetPasswordRequestDao = $userResetPasswordRequestDao;
    }


    /**
     * prihlasi uzivatele nebo vyhodi vyjimku proc to neni mozne
     * @param string $login
     * @param string $password
     */
    public function loginUser($login, $password)
    {
        // TODO nepouzivat global
        /** @var User $user */
        $user = $this->userDao->findUserByLogin($login)->uniqueResult();
        if ($user == null) {
            throw new Exception("Uživatel $login neexistuje.", 100);
        }
        if (!$this->isPasswordCorrect($user, $password)) {
            throw new Exception("Pro uživatele $login nesouhlasí heslo.", 200);
        }
        if (!$user->getActive()) {
            throw new Exception("Uživatel $login není povolený.", 300);
        }

        $user->setLastLoginDate(new Date());
        $this->userDao->update($user);

        // TODO: asi kontrola ze je uz nekdo prihlasen
        $sessionNotStarted = session_status() == PHP_SESSION_NONE;
        if ($sessionNotStarted) { // jeste nebyla nastartovana sessiona, nastartovat
            session_start();
        }
        $_SESSION[$this->userSessionName] = $user;
        if ($sessionNotStarted) { // jeste nebyla nastartovana sessiona, uzavrit ji
            session_write_close();
        }
    }

    /**
     * zkontroluje jestli zadane heslo odpovida zadanemu
     * @param User $user
     */
    protected function isPasswordCorrect(User $user, $password)
    {
        return password_verify($password, $user->getPassword());
    }

    /**
     * odhlasi v soucasnosti prihlaseneho uzivatele
     */
    public function logoutUser()
    {
        $sessionNotStarted = session_status() == PHP_SESSION_NONE;
        if ($sessionNotStarted) { // jeste nebyla nastartovana sessiona, nastartovat
            session_start();
        }
        unset($_SESSION[$this->userSessionName]);
        if ($sessionNotStarted) { // jeste nebyla nastartovana sessiona, uzavrit ji
            session_write_close();
        }
    }

    public function setLoggedUser(User $user)
    {
        if ($user == null) {
            throw new Exception("User can't be null");
        }

        $user = $this->userDao->load($user->getId());

        //
        if (!$user->getActive()) {
            throw new Exception("Uživatele " . $user->getLogin() . " není aktivní.", 300);
        }

        $user->setLastLoginDate(new Date());
        $this->userDao->update($user);

        $sessionNotStarted = session_status() == PHP_SESSION_NONE;
        if ($sessionNotStarted) { // jeste nebyla nastartovana sessiona, nastartovat
            session_start();
        }
        $_SESSION[$this->userSessionName] = $user;
        if ($sessionNotStarted) { // jeste nebyla nastartovana sessiona, uzavrit ji
            session_write_close();
        }
    }

    public function changeLogedUserPassword($password)
    {
        if (!$this->isUserLoggedIn()) {
            throw new Exception("User isn't logged in");
        }

        $user = $this->getLoggedUser();

        $user = $this->userDao->load($user->getId());
        $user->setPassword($this->getPasswordHash($password));

        $this->userDao->update($user);
    }

    public function isUserLoggedIn()
    {
        $sessionNotStarted = session_status() == PHP_SESSION_NONE;
        if ($sessionNotStarted) { // jeste nebyla nastartovana sessiona, nastartovat
            session_start();
        }
        $retval = isset($_SESSION[$this->userSessionName]);
        if ($sessionNotStarted) { // jeste nebyla nastartovana sessiona, uzavrit ji
            session_write_close();
        }
        return $retval;
    }

    public function getLoggedUser()
    {
        $sessionNotStarted = session_status() == PHP_SESSION_NONE;
        if ($sessionNotStarted) { // jeste nebyla nastartovana sessiona, nastartovat
            session_start();
        }
        if (!isset($_SESSION[$this->userSessionName])) {
            throw new Exception("User isn't logged in");
        }
        /** @var User $loggedUser */
        $loggedUser = $_SESSION[$this->userSessionName];
        if ($sessionNotStarted) { // jeste nebyla nastartovana sessiona, uzavrit ji
            session_write_close();
        }

        $roles = $this->roleDao->findUserRoles($loggedUser->getId())->listResult();
        $loggedUser->setRoles($roles);

        $logedUserInfo = new UserInfo($loggedUser);

        return $logedUserInfo;
    }

    public function getPasswordHash($password)
    {
        // use blowfish hash function with cost 10 ai. 10^10 iterations
        $options = [
            'cost' => 10,
        ];
        $hash = password_hash($password, PASSWORD_BCRYPT, $options);

        return $hash;
    }

    public function isLoggedPasswordCorrect($password)
    {
        if (!$this->isUserLoggedIn()) {
            throw new Exception("User isn't logged in");
        }

        $user = $this->getLoggedUser();
        $user = $this->userDao->load($user->getId());

        return password_verify($password, $user->getPassword());
    }

    public function resetUserPassword($userId, $hash, $password)
    {
        $user = $this->userDao->load($userId);
        $user->setPassword($this->getPasswordHash($password));

        $this->userDao->update($user);

        $resets = $this->userResetPasswordRequestDao->find(["fkUserId" => $userId])->listResult();
        foreach ($resets as $reset) {
            $this->userResetPasswordRequestDao->delete($reset);
        }
    }

    public function handleForgottenPasswordEmail($email)
    {
        $user = $this->userDao->find(["email" => $email])->setMaxResults(1)->uniqueResult();

        if ($user == null) {
            throw new Exception();
        }

        $resetPasswordId = sha1($user->getLogin() . "|" . $email . "|" . $user->getName() . "|" . time());

        $expireTime = time() + (6 * 60 * 60); // platne dalsich 6 hodin
        $resetValidUntil = new Date($expireTime);

        $resetRequest = new UserResetPasswordRequest();
        $resetRequest->setResetPasswordId($resetPasswordId);
        $resetRequest->setResetValidUntil($resetValidUntil);
        $resetRequest->setFkUserId($user->getId());

        $mail = $this->generateResetPasswordEmail($user, $resetPasswordId);
        $this->mailer->sendMail($mail);

        // ulozit teprve potom, co se odesle korektne email
        $this->userResetPasswordRequestDao->save($resetRequest);
    }

    abstract public function generateResetPasswordEmail(User $user, $resetPasswordId);

    public function getResetPasswordUser($email, $resetPasswordId)
    {
        $user = $this->userDao->find(["login" => $email])->uniqueResult();
        /** @var User $user */
        if ($user == null) {
            return null;
        }

        $resetPasswordRequests = $this->userResetPasswordRequestDao->findByUserIdResetPasswordId(
            $user->getId(),
            $resetPasswordId
        )->listResult();

        $now = new Date();
        if (count($resetPasswordRequests) > 0
            && $now->before($resetPasswordRequests[0]->getResetValidUntil())) { // test jestli je jeste reset hesla povolen
            return $user;
        }
        return null;
    }

    public function getMinPasswordLength()
    {
        return $this->minPasswordLength;
    }
}
