<?php

namespace IZON\Users\Services\Impl;

use \Exception;

use IZON\Utils\Date;

use IZON\Mailer\Mailer;
use IZON\Mailer\Mail;

use IZON\Users\Services\SessionService;

use IZON\Users\Dao\UserDao;
use IZON\Users\Dao\RoleDao;
use IZON\Users\Dao\UserRoleDao;
use IZON\Users\Dao\UserResetPasswordRequestDao;

use \IZON\Users\Domain\User;

use IZON\Users\Domain\UserResetPasswordRequest;

/**
 * Servis provadejici ostavu session ziskavane z db AdminUser
 */
abstract class SessionServiceImpl implements SessionService {

    /**
     * @var Mailer mailer pro odesilani emailu
     */
    protected $mailer;

    /**
     * @var UserDao dao pro pristup k uzivatelum
     */
    protected $userDao;

    /**
     * @var RoleDao
     */
    protected $roleDao;

    /**
     * @var UserRoleDao
     */
    protected $userRoleDao;

    /**
     * @var UserResetPasswordRequestDao
     */
    protected $userResetPasswordRequestDao;


    /**
     * pod jakym jmenem je uzivatel ulozen v session
     * @var
     */
    protected $userSessionName = "logedUser";

    /**
     * minimalni delka hesla
     * @var integer
     */
    protected $minPasswordLength = 5;


/// generovany konstruktor
    function __construct(Mailer $mailer, UserDao $userDao, RoleDao $roleDao, UserRoleDao $userRoleDao, UserResetPasswordRequestDao $userResetPasswordRequestDao) {
        $this->mailer = $mailer;
        $this->userDao = $userDao;
        $this->roleDao = $roleDao;
        $this->userRoleDao = $userRoleDao;
        $this->userResetPasswordRequestDao = $userResetPasswordRequestDao;
    }


    /**
     * prihlasi uzivatele nebo vyhodi vyjimku proc to neni mozne
     * @param string $login
     * @param string $password
     */
    public function loginUser($login, $password) {
        // TODO nepouzivat global
        /* @var $user User */
        $user = $this->userDao->findUserByLogin($login)->uniqueResult();
        if( $user == NULL) {
            throw new Exception("Uživatel $login neexistuje.", 100);
        }
        if( !$this->isPasswordCorrect($user, $password) ) {
            throw new Exception("Pro uživatele $login nesouhlasí heslo.", 200);
        }
        if( !$user->getActive() ) {
            throw new Exception("Uživatel $login není povolený.", 300);
        }

        $user->setLastLoginDate(new Date());
        $this->userDao->update($user);

        // TODO: asi kontrola ze je uz nekdo prihlasen
        $sessionNotStarted = session_status() == PHP_SESSION_NONE;
        if( $sessionNotStarted ) { // jeste nebyla nastartovana sessiona, nastartovat
            session_start();
        }
        $_SESSION[$this->userSessionName] = $user;
        if( $sessionNotStarted ) { // jeste nebyla nastartovana sessiona, uzavrit ji
            session_write_close();
        }
    }

    /**
     * odhlasi v soucasnosti prihlaseneho uzivatele
     */
    public function logoutUser() {
        $sessionNotStarted = session_status() == PHP_SESSION_NONE;
        if( $sessionNotStarted ) { // jeste nebyla nastartovana sessiona, nastartovat
            session_start();
        }
        unset($_SESSION[$this->userSessionName]);
        if( $sessionNotStarted ) { // jeste nebyla nastartovana sessiona, uzavrit ji
            session_write_close();
        }
    }
    
    public function isUserLoggedIn() {
        $sessionNotStarted = session_status() == PHP_SESSION_NONE;
        if( $sessionNotStarted ) { // jeste nebyla nastartovana sessiona, nastartovat
            session_start();
        }
        $retval = isset($_SESSION[$this->userSessionName]);
        if( $sessionNotStarted ) { // jeste nebyla nastartovana sessiona, uzavrit ji
            session_write_close();
        }
        return $retval;
    }

    public function getLoggedUser() {
       $sessionNotStarted = session_status() == PHP_SESSION_NONE;
        if( $sessionNotStarted ) { // jeste nebyla nastartovana sessiona, nastartovat
            session_start();
        }
        if( !isset($_SESSION[$this->userSessionName]) ) {
            throw new Exception("User isn't logged in");
        }
        /* @var $loggedUser User */
        $loggedUser = $_SESSION[$this->userSessionName];
        if( $sessionNotStarted ) { // jeste nebyla nastartovana sessiona, uzavrit ji
            session_write_close();
        }

        $roles = $this->roleDao->findUserRoles($loggedUser->getId())->listResult();
        $loggedUser->setRoles($roles);

        return $loggedUser;
    }

    public function setLoggedUser(User $user) {
        if( $user == NULL) {
            throw new Exception("User can't be null");
        }

        $user = $this->userDao->load($user->getId());

        //
        if( !$user->getActive() ) {
            throw new Exception("Uživatele ". $user->getLogin() ." není aktivní.", 300);
        }

        $user->setLastLoginDate(new Date());
        $this->userDao->update($user);

        $sessionNotStarted = session_status() == PHP_SESSION_NONE;
        if( $sessionNotStarted ) { // jeste nebyla nastartovana sessiona, nastartovat
            session_start();
        }
        $_SESSION[$this->userSessionName] = $user;
        if( $sessionNotStarted ) { // jeste nebyla nastartovana sessiona, uzavrit ji
            session_write_close();
        }
    }

    public function changeLogedUserPassword($password) {
        if( !$this->isUserLoggedIn() ) {
            throw new Exception("User isn't logged in");
        }

        $user = $this->getLoggedUser();

        $user = $this->userDao->load($user->getId());
        $user->setPassword( $this->getPasswordHash($password) );

        $this->userDao->update($user);
    }

    public function getPasswordHash($password) {
        // use blowfish hash function with cost 10 ai. 10^10 iterations
        $options = [
            'cost' => 10,
        ];
        $hash = password_hash($password, PASSWORD_BCRYPT, $options);

        return $hash;
    }

    public function isLoggedPasswordCorrect($password) {
        if( !$this->isUserLoggedIn() ) {
            throw new Exception("User isn't logged in");
        }

        $user = $this->getLoggedUser();
        $user = $this->userDao->load($user->getId());

        return password_verify($password, $user->getPassword());
    }

    public function resetUserPassword($userId, $hash, $password) {
        $user = $this->userDao->load($userId);
        $user->setPassword($this->getPasswordHash($password));

        $this->userDao->update($user);

        $resets = $this->userResetPasswordRequestDao->find(["fkUserId" => $userId])->listResult();
        foreach($resets as $reset) {
            $this->userResetPasswordRequestDao->delete($reset);
        }
    }

    public function handleForgottenPasswordEmail($email) {
        $user = $this->userDao->find(["email" => $email])->setMaxResults(1)->uniqueResult();

        if( $user == NULL ) {
            throw new \Exception();
        }

        $resetPasswordId = sha1($user->getLogin() ."|". $email ."|". $user->getName() ."|". time() );

        $expireTime = time() + (6 * 60 * 60); // platne dalsich 6 hodin
        $resetValidUntil = new Date($expireTime);

        $resetRequest = new UserResetPasswordRequest();
        $resetRequest->setResetPasswordId($resetPasswordId);
        $resetRequest->setResetValidUntil($resetValidUntil);
        $resetRequest->setFkUserId($user->getId());

        $mail = $this->generateResetPasswordEmail($user, $resetPasswordId);
        $this->mailer->sendMail($mail);
        
        // ulozit teprve potom, co se odesle korektne email
        $this->userResetPasswordRequestDao->save($resetRequest);
    }
    
    public function getResetPasswordUser($email, $resetPasswordId) {
        $user = $this->userDao->find(["login" => $email])->uniqueResult();
        /* @var $user User */
        if( $user == NULL) {
            return NULL;
        }
        
        $resetPasswordRequests = $this->userResetPasswordRequestDao->findByUserIdResetPasswordId($user->getId(), 
                                                                                                 $resetPasswordId)->listResult();
        
        $now = new Date();
        if( count($resetPasswordRequests) > 0
            && $now->before($resetPasswordRequests[0]->getResetValidUntil()) ) { // test jestli je jeste reset hesla povolen
            return $user;
        }
        return NULL;
    }
    
    function getMinPasswordLength() {
        return $this->minPasswordLength;
    }
    
    /**
     * zkontroluje jestli zadane heslo odpovida zadanemu
     * @param User $user
     */
    protected function isPasswordCorrect(User $user, $password) {
        return password_verify($password , $user->getPassword());
    }
    
    public abstract function generateResetPasswordEmail(User $user, $resetPasswordId);
}