<?php

namespace IZON\Thumber\ImageConverter;

use Intervention\Image\Image as InterventionImage;
use Intervention\Image\ImageManager;
use IZON\IO\Image;
use IZON\IO\RasterImage;
use IZON\IO\RasterImageInterface;
use IZON\Thumber\Image\IImagePoint;
use IZON\Thumber\Image\IImageSize;
use IZON\Thumber\Services\ThumbnailServiceInterface;
use IZON\Thumber\Thumbnailer;

/**
 * Description of ImagineImageConverter
 *
 * @author lukas
 */
class ImagineImageConverter implements IImageConverter {

    /**
     * @var InterventionImage
     */
    protected $image;

    /**
     * @var string
     */
    protected $tmpDir;

    /**
     * @var ImageManager
     */
    protected $imageManager;

    /**
     * @var int
     */
    protected $quality;

    /**
     * @var null|string
     */
    protected $format = null;


    /**
     * @param InterventionImage $image
     * @param string $tmpDir
     */
    function __construct(InterventionImage $image, string $tmpDir, ImageManager $imageManager) {
        $this->image = $image;
        $this->tmpDir = $tmpDir;
        $this->imageManager = $imageManager;
        $this->quality = Thumbnailer::DEFAULT_QUALITY;
    }

    public function convert(string $fileName): RasterImageInterface {
        $pathInfo = pathinfo($fileName);

        // uniqid() is used to prevent file name collisions
        $newFileName = $this->tmpDir . (mb_substr($this->tmpDir, -1) != '/' ? '/' : '') . uniqid() .'-'. $pathInfo['basename'];

        $this->image->save($newFileName, $this->quality, $this->format);

        return new RasterImage($newFileName);
    }

    /**
     * set forced format
     * @param string $format
     * @return IImageConverter
     */
    public function setFormat(string $format): IImageConverter {
        $this->format = $format;
        return $this;
    }

    /**
     *
     * @param int $quality
     * @return IImageConverter
     */
    public function setQuality(int $quality): IImageConverter {
        $this->quality = $quality;
        return $this;
    }

    /**
     *
     * @param IImageSize $imageSize
     * @return IImageConverter
     */
    public function setSize(IImageSize $imageSize): IImageConverter {
        $this->image->resize($imageSize->getWidth(), $imageSize->getHeight());
        return $this;
    }

    /**
     * @inheritDoc
     */
    public function crop(IImageSize $size, IImagePoint $point): IImageConverter {
        $this->image->crop($size->getWidth(), $size->getHeight(), $point->getX(), $point->getY());
        return $this;
    }

    /**
     * @inheritDoc
     */
    public function setInterlace(bool $interlace): IImageConverter {
        $this->image->interlace($interlace);
        return $this;
    }

    /**
     * @inheritDoc
     */
    public function watermark(
        RasterImageInterface $watermarkImage,
        IImageSize $watermarkSize,
        string $alignment,
        IImagePoint $imagePosition
    ): IImageConverter {
        /** @var InterventionImage $watermarkImage */
        $watermarkImage = $this->imageManager->make($watermarkImage->getFileStream());
        $watermarkImage->resize($watermarkSize->getWidth(), $watermarkSize->getHeight());

        $this->image->insert($watermarkImage, $alignment, $imagePosition->getX(), $imagePosition->getY());
        return $this;
    }

    /**
     * @inheritDoc
     */
    public function destroyImage(RasterImageInterface $convertedFile): void {
        unlink($convertedFile->getRealPath());
    }
}
