<?php

namespace IZON\Thumber\ImageConfiguration;

use IZON\IO\RasterImageInterface;
use IZON\Thumber\Image\IImagePoint;
use IZON\Thumber\Image\IImageSize;

/**
 * ImageConfiguration
 * @author Aleš Kopecký <kopecky@izon.cz>
 */
class ImageConfiguration implements IImageConfiguration
{
    protected RasterImageInterface $image;

    protected ?IImageSize $imageSize = null;

    protected int $quality;

    protected string $type;

    protected ?string $format = null;

    /**
     *
     * @var IImageSize|null
     */
    protected ?IImageSize $cropSize = null;

    /**
     * @var IImagePoint|null
     */
    protected ?IImagePoint $cropPoint = null;

    /**
     * @var bool if true resized image will be cached on server
     */
    protected bool $cache = true;

    /**
     * @var RasterImageInterface|null watermark configuration
     */
    protected ?RasterImageInterface $watermarkImage = null;

    /**
     * @var IImageSize|null
     */
    protected ?IImageSize $watermarkSize = null;

    /**
     * @var string|null
     */
    protected ?string $watermarkAlignment = null;

    /**
     * @var IImagePoint|null
     */
    protected ?IImagePoint $watermarkPosition = null;

    /**
     * @var string|int|null
     */
    protected $watermark = null;

    protected bool $lazy = false;


    /**
     *
     * @return RasterImageInterface
     */
    public function getImage(): RasterImageInterface
    {
        return $this->image;
    }

    /**
     *
     * @param RasterImageInterface $image
     * @return $this
     */
    public function setImage(RasterImageInterface $image): self
    {
        $this->image = $image;
        return $this;
    }

    /**
     * {@inheritDoc}
     */
    public function getImageSize(): ?IImageSize
    {
        return $this->imageSize;
    }

    /**
     *
     * @param IImageSize $imageSize
     * @return $this
     */
    public function setImageSize(IImageSize $imageSize)
    {
        $this->imageSize = $imageSize;
        return $this;
    }

    /**
     *
     * @return int|null
     */
    public function getWidth(): ?int
    {
        if (!$this->imageSize instanceof IImageSize) {
            return null;
        }
        return $this->imageSize->getWidth();
    }

    /**
     *
     * @return int|null
     */
    public function getHeight(): ?int
    {
        if (!$this->imageSize instanceof IImageSize) {
            return null;
        }
        return $this->imageSize->getHeight();
    }

    /**
     *
     * @return int
     */
    public function getQuality(): int
    {
        return $this->quality;
    }

    /**
     *
     * @param int $quality
     * @return $this
     */
    public function setQuality(int $quality)
    {
        $this->quality = $quality;
        return $this;
    }

    /**
     *
     * @return string
     */
    public function getType(): string
    {
        return $this->type;
    }

    /**
     *
     * @param string $type
     * @return $this
     */
    public function setType(string $type)
    {
        $this->type = $type;
        return $this;
    }

    /**
     *
     * @return IImageSize|null
     */
    public function getCropSize(): ?IImageSize
    {
        return $this->cropSize;
    }

    /**
     *
     * @param IImageSize $cropSize
     * @return $this
     */
    public function setCropSize(IImageSize $cropSize)
    {
        $this->cropSize = $cropSize;
        return $this;
    }

    /**
     *
     * @return IImagePoint|null
     */
    public function getCropPoint(): ?IImagePoint
    {
        return $this->cropPoint;
    }

    /**
     *
     * @param IImagePoint $cropPoint
     * @return $this
     */
    public function setCropPoint(IImagePoint $cropPoint)
    {
        $this->cropPoint = $cropPoint;
        return $this;
    }

    /**
     *
     * @return bool
     */
    public function hasCrop(): bool
    {
        return $this->cropPoint instanceof IImagePoint && $this->cropSize instanceof IImageSize;
    }

    public function getOrigin(): ?string
    {
        return $this->image->getOriginUID();
    }

    /**
     * @return RasterImageInterface|null
     */
    public function getWatermarkImage(): ?RasterImageInterface
    {
        return $this->watermarkImage;
    }

    /**
     * @param RasterImageInterface|null $watermarkImage
     */
    public function setWatermarkImage(?RasterImageInterface $watermarkImage): IImageConfiguration
    {
        $this->watermarkImage = $watermarkImage;
        return $this;
    }

    /**
     * @return IImageSize|null
     */
    public function getWatermarkSize(): ?IImageSize
    {
        return $this->watermarkSize;
    }

    /**
     * @param IImageSize|null $watermarkSize
     * @return $this
     */
    public function setWatermarkSize(?IImageSize $watermarkSize): IImageConfiguration
    {
        $this->watermarkSize = $watermarkSize;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getWatermarkAlignment(): ?string
    {
        return $this->watermarkAlignment;
    }

    /**
     * @param string|null $watermarkAlignment
     * @return $this
     */
    public function setWatermarkAlignment(?string $watermarkAlignment): IImageConfiguration
    {
        $this->watermarkAlignment = $watermarkAlignment;
        return $this;
    }

    /**
     * @return IImagePoint|null
     */
    public function getWatermarkPosition(): ?IImagePoint
    {
        return $this->watermarkPosition;
    }

    /**
     * @param IImagePoint|null $watermarkPosition
     * @return $this
     */
    public function setWatermarkPosition(?IImagePoint $watermarkPosition): IImageConfiguration
    {
        $this->watermarkPosition = $watermarkPosition;
        return $this;
    }

    /**
     * {@inheritDoc}
     */
    public function isCache(): bool
    {
        return $this->cache;
    }

    /**
     * @param bool $cache
     * @return ImageConfiguration
     */
    public function setCache(bool $cache): IImageConfiguration
    {
        $this->cache = $cache;
        return $this;
    }

    /**
     * @return bool
     */
    public function isLazy(): bool
    {
        return $this->lazy;
    }

    /**
     * @param bool $lazy
     * @return $this
     */
    public function setLazy(bool $lazy): IImageConfiguration
    {
        $this->lazy = $lazy;
        return $this;
    }

    /**
     * @return null|string
     */
    public function getFormat(): ?string
    {
        return $this->format;
    }

    /**
     * @param null|string $format
     */
    public function setFormat(?string $format): IImageConfiguration
    {
        $this->format = $format;
        return $this;
    }

    /**
     * @return int|string|null
     */
    public function getWatermark()
    {
        return $this->watermark;
    }

    /**
     * @param int|string|null $watermark
     * @return IImageConfiguration
     */
    public function setWatermark($watermark): IImageConfiguration
    {
        $this->watermark = $watermark;
        return $this;
    }
}
