<?php

namespace IZON\Thumber\Web\Controllers;

use Exception;
use finfo;
use InvalidArgumentException;
use IZON\MVC\Controller;
use IZON\MVC\Exceptions\BadRequestException;
use IZON\MVC\Exceptions\InternalServerErrorException;
use IZON\MVC\Exceptions\PageNotFoundException;
use IZON\MVC\Messages\HttpRequestInterface;
use IZON\MVC\ModelAndView;
use IZON\Thumber\Exceptions\FilesystemNotFoundException;
use IZON\Thumber\Exceptions\ImageNotFoundException;
use IZON\Thumber\Exceptions\ImageSizeIsNotSetException;
use IZON\Thumber\Exceptions\InvalidResizerStrategyException;
use IZON\Thumber\Exceptions\UnsupportedImageFormatException;
use IZON\Thumber\Exceptions\UnsupportedResizeTypeException;
use IZON\Thumber\Image\Exceptions\InvalidSizeException;
use IZON\Thumber\Services\ThumbnailServiceInterface;
use IZON\Thumber\Web\Utils\WebPImageUtils;
use IZON\Utils\Slug;

/**
 * Controller which handles lazy creating images in cache.
 */
class ImageLazyController implements Controller
{
    protected ThumbnailServiceInterface $thumbnailService;

    protected bool $enableWebp = true;

    /**
     *
     * @param ThumbnailServiceInterface $thumberService
     */
    public function __construct(ThumbnailServiceInterface $thumberService)
    {
        $this->thumbnailService = $thumberService;
    }

    /**
     * Based on parameters in URL creates demanded image in cache and redirects to it.
     * @param HttpRequestInterface $request
     * @return ModelAndView
     */
    public function execute(HttpRequestInterface $request)
    {
        if (
            $this->enableWebp
            && WebPImageUtils::hasWebPSupport(getallheaders())
        ) {
            // Do something for WebP users
            $this->thumbnailService->setFormat('webp');
        }

        try {
            $image = $this->thumbnailService
                ->getImageURLBuilder()
                ->getImage($request->getQueryParams());
            $imageConfiguration = $this->thumbnailService
                ->createThumbnailImageConfiguration($image, $request->getQueryParams());
        } catch (ImageNotFoundException $ex) {
            throw new PageNotFoundException($ex->getMessage(), $ex->getCode(), $ex);
        } catch (FilesystemNotFoundException $ex) {
            throw new PageNotFoundException($ex->getMessage(), $ex->getCode(), $ex);
        } catch (InvalidArgumentException $ex) {
            throw new BadRequestException("Some required parameters not provided", $ex->getCode(), $ex);
        } catch (ImageSizeIsNotSetException $ex) {
            throw new BadRequestException("Not set size of image", $ex->getCode(), $ex);
        } catch (UnsupportedImageFormatException $ex) {
            throw new BadRequestException("Unsupported image format", $ex->getCode(), $ex);
        } catch (InvalidResizerStrategyException $ex) {
            throw new BadRequestException("Invalid resize strategy", $ex->getCode(), $ex);
        } catch (UnsupportedResizeTypeException $ex) {
            throw new BadRequestException("Unsupported resize type", $ex->getCode(), $ex);
        } catch (InvalidSizeException $ex) {
            throw new BadRequestException("Unsupported resize size", $ex->getCode(), $ex);
        } catch (Exception $ex) {
            throw new InternalServerErrorException("Something went wrong", $ex->getCode(), $ex);
        }

        // this is call for thumbnail creation
        $filePath = $this->thumbnailService->createThumbnail($imageConfiguration);
        if (!$imageConfiguration->isCache()) {
            $finfo = new finfo(FILEINFO_MIME_TYPE);
            $mime = $finfo->buffer($filePath);
            $ext = str_replace('image/', '', $mime);
            $filename = Slug::createSlug($image->getFileName()) . '.' . $ext;
            header('Content-Type: ' . $mime);
            header('Content-Disposition: attachment; filename="' . $filename . '"');
            echo $filePath;
            die;
        }
        return ModelAndView::createURLRedirect($filePath);
    }
}
