<?php

/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */

namespace IZON\Thumber\Image;

use IZON\IO\FileInterface;
use IZON\Thumber\ImageResizer\IImageResizer;
use IZON\Thumber\Services\ThumbnailServiceInterface;
use IZON\Thumber\Thumbnailer;
use function IZON\File\normalizePath;

/**
 * Description of ImageNameBuilder
 *
 * @author lukas
 */
class ImageNameBuilder implements IImageNameBuilder {

    protected string $name;

    protected string $path;

    protected ?string $format = null;

    protected ?IImageSize $size = null;

    protected ?int $quality = null;

    protected string $type = IImageResizer::CONTAIN;

    protected ?IImagePoint $cropPoint = null;

    protected ?IImageSize $cropSize = null;

    protected ?string $origin = null;

    /**
     * ImageNameBuilder constructor.
     * @param FileInterface $file
     */
    public function __construct(FileInterface $file) {
        $this->name = $file->getFileName();
        $this->path = $file->getFileUID();
        $this->format = $file->getFileExtension();
        $this->origin = $file->getOriginUID();
    }

    /**
     *
     * @return string
     */
    public function build(): string {
        $filename = '';
        if(!empty($this->origin)) {
            // store images to folders by origin to make rm -rf on image-cache easier
            $filename = $this->origin .'/';
        }

        // clean empty path parts
        $pathParts = explode('/', normalizePath($this->path));
        $pathParts = array_values(array_filter($pathParts));
        // remove filename from path if it is the last part
        if($pathParts[count($pathParts) - 1] === $this->name) {
            array_pop($pathParts);
        }
        $filename .= implode('/', $pathParts);

        $filename .= '/' . $this->name;
        if(!empty($this->size)) {
            $filename .= '_' . $this->size->getWidth() . 'x' . $this->size->getHeight();
        }
        if(!is_null($this->quality) && $this->quality != Thumbnailer::DEFAULT_QUALITY) {
            $filename .= '-q' . $this->quality;
        }
        if($this->type !== IImageResizer::CONTAIN) {
            $filename .= '-'.$this->type;
        }
        if(!empty($this->cropPoint)) {
            $filename .= '-cp_'.$this->cropPoint->getX().'x'.$this->cropPoint->getY();
        }
        if(!empty($this->cropSize)) {
            $filename .= '-cs_'. $this->cropSize->getWidth() . 'x' . $this->cropSize->getHeight();
        }
        if(!empty($this->format)) {
            $filename .= '.'. $this->format;
        }
        return $filename;
    }

    /**
     *
     * @param string $format
     * @return IImageNameBuilder
     */
    public function setFormat(string $format): IImageNameBuilder {
        $this->format = $format;
        return $this;
    }

    /**
     *
     * @param int $quality
     * @return IImageNameBuilder
     */
    public function setQuality(int $quality): IImageNameBuilder {
        $this->quality = $quality;
        return $this;
    }

    /**
     *
     * @param IImageSize $size
     * @return IImageNameBuilder
     */
    public function setSize(IImageSize $size): IImageNameBuilder {
        $this->size = $size;
        return $this;
    }

    /**
     *
     * @param string $type
     * @return IImageNameBuilder
     */
    public function setType(string $type): IImageNameBuilder {
        $this->type = $type;
        return $this;
    }
    /**
     *
     * @param IImagePoint $point
     * @return IImageNameBuilder
     */
    public function setCropPoint(IImagePoint $point): IImageNameBuilder {
        $this->cropPoint = $point;
        return $this;
    }

     /**
     *
     * @param IImageSize $size
     * @return IImageNameBuilder
     */
    public function setCropSize(IImageSize $size): IImageNameBuilder {
        $this->cropSize = $size;
        return $this;
    }

}
