<?php

namespace IZON\Thumber\ImageCache;

use IZON\Filesystem\FilesystemInterface;
use IZON\IO\Image;
use IZON\IO\RasterImageInterface;
use IZON\Thumber\Exceptions\ImageNotInCacheException;
use IZON\Thumber\Exceptions\ThumbnailException;

/**
 * Description of ImageCache
 *
 * @author lukas
 */
class ImageCache {

    protected FilesystemInterface $filesystem;

    protected string $dir;

    /**
     * ImageCache constructor.
     * @param FilesystemInterface $filesystem
     */
    public function __construct(FilesystemInterface $filesystem, string $dir) {
        $this->filesystem = $filesystem;
        $this->dir = $dir;
    }


    /**
     *
     * @param string $filename
     * @return bool
     */
    public function has(string $filename): bool {
      return $this->filesystem->has($filename);
    }

    /**
     *
     * @param string $filename
     * @return RasterImageInterface
     * @throws ImageNotInCacheException
     * @throws ThumbnailException
     */
    public function get(string $filename): RasterImageInterface {
        if(!$this->filesystem->has($filename)) {
            throw new ImageNotInCacheException();
        }
        $file = $this->filesystem->getFile($filename);
        if(!$file instanceof RasterImageInterface) {
            throw new ThumbnailException("File '. $filename .' is not instance of RasterImageInterface");
        }
        return $file;
    }

    /**
     *
     * @param string $filename returns url path where parts are url encoded
     * @return string
     * @throws ImageNotInCacheException
     */
    public function getPath(string $filename): string {
        if(!$this->filesystem->has($filename)) {
            throw new ImageNotInCacheException();
        }

        $pathParts = explode('/', $this->getFilePath($filename));
        $pathParts = array_map(fn($part) => rawurlencode($part), $pathParts);
        return "/". implode('/', $pathParts);
    }

    /**
     * returns cache image timestamp
     * @param string $filename
     * @return int
     */
    public function getTimestamp(string $filename): int {
        if( !$this->filesystem->has($filename) ) {
            throw new ImageNotInCacheException();
        }
        return $this->filesystem->getTimestamp($filename);
    }

    /**
     *
     * @param string $filename
     * @param Image $image
     */
    public function set(string $filename, RasterImageInterface $image): void {
        $this->filesystem->putStream($filename, $image->getFileStream());
    }

    protected function getFilePath($filename): string {
        return $this->dir . (mb_substr($this->dir, -1) != '/' ? '/' : '') . $filename;
    }

}
