<?php

namespace IZON\Thumber\ImageConfiguration;

use IZON\IO\Image;
use IZON\IO\RasterImageInterface;
use IZON\Thumber\Image\IImagePoint;
use IZON\Thumber\Image\IImageSize;

/**
 * ImageConfiguration
 * @author Aleš Kopecký <kopecky@izon.cz>
 */
class ImageConfiguration implements IImageConfiguration{

    /**
     *
     * @var RasterImageInterface
     */
    protected $image;

    /**
     * @var bool
     */
    protected $lazy;

    /**
     * @var IImageSize
     */
    protected $imageSize;

    /**
     * @var int
     */
    protected $quality;

    /**
     * @var string
     */
    protected $type;

    /**
     * @var string name of resized image
     */
    protected $name;

    /**
     * @var null|string image file format to use for thumbnail a.i. jpg, png, ...
     */
    protected $format;

    /**
     * @var IImageSize|null
     */
    protected $cropSize;

    /**
     * @var IImagePoint|null
     */
    protected $cropPoint;

    /**
     * @var RasterImageInterface|null watermark configuration
     */
    protected $watermarkImage;

    /**
     * @var IImageSize|null
     */
    protected $watermarkSize;

    /**
     * @var string|null
     */
    protected $watermarkAlignment;

    /**
     * @var IImagePoint|null
     */
    protected $watermarkPosition;


    /**
     *
     * @return RasterImageInterface
     */
    public function getImage(): RasterImageInterface {
        return $this->image;
    }

    /**
     *
     * @return IImageSize
     */
    public function getImageSize(): IImageSize {
        return $this->imageSize;
    }

    /**
     *
     * @return int|null
     */
    public function getWidth(): ?int {
        if(!$this->imageSize instanceof IImageSize) {
            return null;
        }
        return $this->imageSize->getWidth();
    }

    /**
     *
     * @return int|null
     */
    public function getHeight(): ?int {
        if(!$this->imageSize instanceof IImageSize) {
            return null;
        }
        return $this->imageSize->getHeight();
    }

    /**
     *
     * @return int
     */
    public function getQuality(): int {
        return $this->quality;
    }

    /**
     *
     * @return string
     */
    public function getType(): string {
        return $this->type;
    }

    /**
     *
     * @return string
     */
    public function getName(): string {
        return $this->name;
    }

    /**
     *
     * @return IImageSize|null
     */
    public function getCropSize(): ?IImageSize {
        return $this->cropSize;
    }

    /**
     *
     * @return IImagePoint|null
     */
    public function getCropPoint(): ?IImagePoint {
        return $this->cropPoint;
    }

    /**
     *
     * @param RasterImageInterface $image
     * @return $this
     */
    public function setImage(RasterImageInterface $image) {
        $this->image = $image;
        return $this;
    }

    /**
     *
     * @param IImageSize $imageSize
     * @return $this
     */
    public function setImageSize(IImageSize $imageSize) {
        $this->imageSize = $imageSize;
        return $this;
    }

    /**
     *
     * @param int $quality
     * @return $this
     */
    public function setQuality(int $quality) {
        $this->quality = $quality;
        return $this;
    }

    /**
     *
     * @param string $type
     * @return $this
     */
    public function setType(string $type) {
        $this->type = $type;
        return $this;
    }
    /**
     *
     * @param string $name
     * @return $this
     */
    public function setName(string $name) {
        $this->name = $name;
        return $this;
    }

    /**
     *
     * @param IImageSize $cropSize
     * @return $this
     */
    public function setCropSize(IImageSize $cropSize) {
        $this->cropSize = $cropSize;
        return $this;
    }

    /**
     *
     * @param IImagePoint $cropPoint
     * @return $this
     */
    public function setCropPoint(IImagePoint $cropPoint) {
        $this->cropPoint = $cropPoint;
        return $this;
    }

    /**
     *
     * @return bool
     */
    public function hasCrop(): bool {
        return ($this->cropPoint instanceof IImagePoint && $this->cropSize instanceof IImageSize);
    }

    public function getOrigin(): ?string {
        return $this->image->getOriginUID();
    }

    /**
     * @return RasterImageInterface|null
     */
    public function getWatermarkImage(): ?RasterImageInterface {
        return $this->watermarkImage;
    }

    /**
     * @param RasterImageInterface|null $watermarkImage
     */
    public function setWatermarkImage(?RasterImageInterface $watermarkImage): IImageConfiguration {
        $this->watermarkImage = $watermarkImage;
        return $this;
    }

    /**
     * @return IImageSize|null
     */
    public function getWatermarkSize(): ?IImageSize {
        return $this->watermarkSize;
    }

    /**
     * @param IImageSize|null $watermarkSize
     * @return $this
     */
    public function setWatermarkSize(?IImageSize $watermarkSize): IImageConfiguration {
        $this->watermarkSize = $watermarkSize;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getWatermarkAlignment(): ?string {
        return $this->watermarkAlignment;
    }

    /**
     * @param string|null $watermarkAlignment
     * @return $this
     */
    public function setWatermarkAlignment(?string $watermarkAlignment): IImageConfiguration {
        $this->watermarkAlignment = $watermarkAlignment;
        return $this;
    }

    /**
     * @return IImagePoint|null
     */
    public function getWatermarkPosition(): ?IImagePoint {
        return $this->watermarkPosition;
    }

    /**
     * @param IImagePoint|null $watermarkPosition
     * @return $this
     */
    public function setWatermarkPosition(?IImagePoint $watermarkPosition): IImageConfiguration {
        $this->watermarkPosition = $watermarkPosition;
        return $this;
    }

    /**
     * @return bool
     */
    public function isLazy(): bool {
        return $this->lazy;
    }

    /**
     * @param bool $lazy
     * @return $this
     */
    public function setLazy(bool $lazy): IImageConfiguration {
        $this->lazy = $lazy;
        return $this;
    }

    /**
     * @return null|string
     */
    public function getFormat(): ?string {
        return $this->format;
    }

    /**
     * @param null|string $format
     */
    public function setFormat(?string $format): IImageConfiguration {
        $this->format = $format;
        return $this;
    }
}
