<?php

namespace IZON\Thumber\Factory;

use IZON\Thumber\Image\IImageUrlBuilder;
use IZON\Thumber\Image\ImageUrlBuilder;
use IZON\Thumber\ImageCache\ImageCache;
use IZON\Thumber\ImageConverter\IImageConverterFactory;
use IZON\Thumber\ImageConverter\ImagineImageCoverterFactory;
use IZON\Thumber\ImageResizer\IImageResizer;
use IZON\Thumber\ImageResizer\ImageResizer;
use IZON\Thumber\ImageResizer\Strategies\ContainImageResizerStrategy;
use IZON\Thumber\ImageResizer\Strategies\CoverImageResizerStrategy;
use IZON\Thumber\ImageResizer\Strategies\CropImageResizerStrategy;
use IZON\Thumber\Services\Impl\ThumbnailServiceImpl;
use IZON\Thumber\Services\ThumbnailService;

/**
 * Description of ThumberServiceFactory
 * create thumber service
 * 
 * @author Lukáš Linhart
 */
class ThumberServiceFactory {
    /**
     *
     * @var string temporary directory
     */
    protected $tmpDir;
    /**
     * 
     * @var string root directory
     */
    protected $rootDir;
    /**
     *
     * @var string driver name
     */
    protected $driver;
    /**
     *
     * @var string cache directory
     */
    protected $cacheDir;
    /**
     *
     * @var string builder url path
     */
    protected $builderPath;
    /**
     * factory construtor
     * 
     * @param string $rootDir root directory
     * @param string $tmpDir tmp directory
     * @param string $cacheDir cache diretory
     * @param string $builderPath url path for builder
     */
    public function __construct(string $rootDir, string $tmpDir, string $cacheDir, string $builderPath) {
        $this->rootDir = $rootDir;
        $this->tmpDir = $tmpDir;
        $this->cacheDir = $cacheDir;
        $this->builderPath = $builderPath;
        $this->driver = 'gd';
    }
    /**
     * set driver for thumbnail service
     * 
     * @param string $driver driver gd|imagemagick
     */
    public function setDriver(string $driver) {
        $this->driver = $driver;
    }
    /**
     * create thumbnail service
     * 
     * @return ThumbnailService
     */
    public function create(): ThumbnailService {
        $imageResizer = $this->createResizer();
        $imageCache = $this->createCache();
        $imageConverterFactory = $this->createConvertor();
        $imageBuilder = $this->createURLBuilder();

        $service = new ThumbnailServiceImpl($imageResizer, $imageCache, $imageConverterFactory);
        $service->setRootDir($this->rootDir);
        $service->setImageUrlBuilder($imageBuilder);
        return $service;
    }
    /**
     * create resizer for thumbnail service
     * 
     * @return IImageResizer
     */
    protected function createResizer(): IImageResizer {
        $strategies = [
            ImageResizer::CONTAIN => new ContainImageResizerStrategy(),
            ImageResizer::COVER => new CoverImageResizerStrategy(),
            ImageResizer::CROP => new CropImageResizerStrategy()
        ];
        $resizer = new ImageResizer($strategies);
        return $resizer;
    }
    /**
     * create image cache for thumbnail service
     * 
     * @return ImageCache
     */
    protected function createCache(): ImageCache {
        $cache = new ImageCache($this->cacheDir);
        return $cache;
    }
    /**
     * create convertor factory for thumbnail service
     * 
     * @return IImageConverterFactory
     */
    protected function createConvertor(): IImageConverterFactory {
        $convertor = new ImagineImageCoverterFactory($this->tmpDir, $this->driver);
        return $convertor;
    }
    /**
     * create image url builder for thumbnail service
     * @return IImageUrlBuilder
     */
    protected function createURLBuilder(): IImageUrlBuilder {
        $builder = new ImageUrlBuilder();
        $builder->setPath($this->builderPath);
        return $builder;
    }

}
