<?php

namespace IZON\Thumber\Web\Controllers;

use IZON\MVC\Controller;
use IZON\MVC\Exceptions\BadRequestException;
use IZON\MVC\Exceptions\InternalServerErrorException;
use IZON\MVC\Exceptions\PageNotFoundException;
use IZON\MVC\HttpRequest;
use IZON\MVC\ModelAndView;
use IZON\Thumber\Exceptions\ImageNotFoundException;
use IZON\Thumber\Exceptions\ImageSizeIsNotSetException;
use IZON\Thumber\Exceptions\InvalidResizerStrategyException;
use IZON\Thumber\Exceptions\UnsupportedImageFormatException;
use IZON\Thumber\Exceptions\UnsupportedImageQualityException;
use IZON\Thumber\Exceptions\UnsupportedResizeTypeException;
use IZON\Thumber\Image\Exceptions\InvalidSizeException;
use IZON\Thumber\Image\ImageUrlBuilder;
use IZON\Thumber\Services\ThumbnailService;
use IZON\Thumber\Web\Utils\WebPImageUtils;

/**
 * Controller which handles lazy creating images in cache.
 */
class ImageLazyController implements Controller {

    protected ThumbnailService $thumberService;

    protected bool $enableWebp = true;

    /**
     * 
     * @param ThumbnailService $thumberService
     */
    public function __construct(ThumbnailService $thumberService) {
        $this->thumberService = $thumberService;
    }
    
    /**
     * Based on parameters in URL creates demanded image in cache and redirects to it.
     * @param HttpRequest $request
     * @return ModelAndView
     */
    public function execute(HttpRequest $request) {
        if( WebPImageUtils::hasWebPSupport(getallheaders()) ) {
            // Do something for WebP users
            $this->thumberService->setFormat('webp');
        }

        try {
            $imageConfiguration = ImageUrlBuilder::fromArray($request->getParameters());
        } catch(ImageNotFoundException $ex) {
            throw new PageNotFoundException("Not found Image", $ex->getCode(), $ex);
        } catch(ImageSizeIsNotSetException $ex) {
            throw new BadRequestException("Not set size of image", $ex->getCode(), $ex);
        } catch(UnsupportedImageFormatException $ex) {
            throw new BadRequestException("Unsupported image format", $ex->getCode(), $ex);
        } catch(UnsupportedResizeTypeException $ex) {
            throw new BadRequestException("Unsupported resize type", $ex->getCode(), $ex);
        } catch(UnsupportedImageQualityException $ex) {
            throw new BadRequestException("Unsupported resize type", $ex->getCode(), $ex);
        } catch(InvalidSizeException $ex) {
            throw new BadRequestException("Invalid image size", $ex->getCode(), $ex);
        } catch(\Exception $ex) {
            throw new InternalServerErrorException("Something went wrong", $ex->getCode(), $ex);
        }
        $filePath = $this->thumberService->createThumbnailPath($imageConfiguration);
        return ModelAndView::createURLRedirect('/'.$filePath);        
    }

    public function setEnableWebp(bool $enableWebp): void {
        $this->enableWebp = $enableWebp;
    }
}
