<?php

namespace IZON\Templating\Latte\Helpers\Tags;

use RuntimeException;

class ViteAssetsAllTagSource extends AbstractUnpairedCallableTagSource
{
    /**
     * @var array<string, array{host: string, buildDir?: string, buildPath?: string, dev: bool}>
     */
    protected array $config;

    /**
     * @var array<string, array<string, mixed>>
     */
    protected array $manifests = [];

    protected string $publicDir;

    /**
     * @param string $publicDir
     * @param array<string, array{host: string, buildDir?: string, buildPath?: string, dev: bool}> $config
     */
    public function __construct(string $publicDir, array $config)
    {
        parent::__construct('viteAssetsAll');
        $this->publicDir = $publicDir;
        $this->config = $config;
    }

    public function getProvider(): callable
    {
        $configs = $this->config;
        $loadManifest = fn (string $instance) => $this->loadManifest($instance);
        return function (string $instance, string $path) use ($configs, $loadManifest): string {
            $config = $configs[$instance] ?? null;
            if ($config === null) {
                throw new RuntimeException("Instance '$instance' not exist in vite config.");
            }
            $buildPath = $config['buildPath'] ?? null;

            if (!$config['dev']) {
                $manifest = $loadManifest($instance);
                $assetInfo = $manifest[$path] ?? null;
                if ($assetInfo === null) {
                    throw new RuntimeException("Path '$path' not exist in manifest.");
                }
                $realPath = $assetInfo['file'];
                $src = "$buildPath/$realPath";
                $sources = [
                    $src
                ];
                foreach ($assetInfo['css'] as $css) {
                    $src = "$buildPath/$css";
                    $sources[] = $src;
                }
                foreach ($assetInfo['imports'] as $js) {
                    $this->getProvider()($instance, $js);
                }
                foreach ($sources as $src) {
                    $this->printSource($src);
                }
            }
            return "";
        };
    }

    protected function printSource(string $src): void
    {
        if (str_ends_with($src, '.css')) {
            echo '<link rel="stylesheet" href="' . $src . '">' . PHP_EOL;
        } else {
            echo '<script type="module" src="' . $src . '"></script>' . PHP_EOL;
        }
    }

    /**
     * @param string $instance
     * @return array<string, mixed>
     */
    protected function loadManifest(string $instance): array
    {
        if (array_key_exists($instance, $this->manifests)) {
            return $this->manifests[$instance];
        }

        $config = $this->config[$instance] ?? null;
        if ($config === null) {
            throw new RuntimeException("Instance '$instance' not exist in vite config.");
        }
        $buildDir = $config['buildDir'] ?? "{$this->publicDir}{$config['buildPath']}";
        $manifestPath = "$buildDir/.vite/manifest.json";
        $exists = file_exists($manifestPath);

        if (!$exists) {
            throw new RuntimeException("File '$manifestPath' not exist.");
        }

        if (is_dir($manifestPath)) {
            throw new RuntimeException("File '$manifestPath' is directory.");
        }

        $content = file_get_contents($manifestPath);
        if ($content === false) {
            throw new RuntimeException("Cannot read file '$manifestPath'.");
        }

        $manifest = json_decode($content, true);
        if (JSON_ERROR_NONE !== $error = json_last_error()) {
            throw new RuntimeException("File '$manifestPath' is not valid json. Error: " . $error);
        }
        // DOTO: validate manifest structure

        $this->manifests[$instance] = $manifest;

        return $manifest;
    }
}
