<?php

declare(strict_types=1);

namespace IZON\Templating\Latte\Helpers\Functions;

use RuntimeException;
use function file_exists;
use function is_dir;

class ViteAssetFunction implements TemplateFunctionInterface
{
    protected string $publicDir;

    /**
     * @var array<string, array{host: string, buildDir: string|null, buildPath: string|null, dev: bool}>
     */
    protected array $config;

    /**
     * @var array<string, array<string, mixed>>
     */
    protected array $manifests = [];

    /**
     * @param array<string, array{host: string, buildDir: string|null, buildPath: string|null, dev: bool}> $config
     */
    public function __construct(string $publicDir, array $config)
    {
        $this->publicDir = $publicDir;
        $this->config = $config;
    }

    public function getName(): string
    {
        return 'viteAsset';
    }

    public function getFunction(): callable
    {
        $configs = $this->config;
        $loadManifest = fn (string $instance) => $this->loadManifest($instance);
        return function (string $instance, string $path) use ($configs, $loadManifest): string {
            $config = $configs[$instance] ?? null;
            if ($config === null) {
                throw new RuntimeException("Instance '$instance' not exist in vite config.");
            }
            $buildPath = $config['buildPath'] ?? null;


            if (!$config['dev']) {
                $manifest = $loadManifest($instance);
                $assetInfo = $manifest[$path] ?? null;
                if ($assetInfo === null) {
                    throw new RuntimeException("Path '$path' not exist in manifest.");
                }
                $realPath = $assetInfo['file'];
                return "$buildPath/$realPath";
            }

            return "$config[host]/$path";
        };
    }

    /**
     * @param string $instance
     * @return array<string, mixed>
     */
    protected function loadManifest(string $instance): array
    {
        if (array_key_exists($instance, $this->manifests)) {
            return $this->manifests[$instance];
        }

        $config = $this->config[$instance] ?? null;
        if ($config === null) {
            throw new RuntimeException("Instance '$instance' not exist in vite config.");
        }
        $buildDir = $config['buildDir'] ?? "{$this->publicDir}{$config['buildPath']}";
        $manifestPath = "$buildDir/.vite/manifest.json";
        $exists = file_exists($manifestPath);

        if (!$exists) {
            throw new RuntimeException("File '$manifestPath' not exist.");
        }

        if (is_dir($manifestPath)) {
            throw new RuntimeException("File '$manifestPath' is directory.");
        }

        $content = file_get_contents($manifestPath);
        if ($content === false) {
            throw new RuntimeException("File '$manifestPath' cannot be read.");
        }

        $manifest = json_decode($content, true);
        if (JSON_ERROR_NONE !== $error = json_last_error()) {
            throw new RuntimeException("File '$manifestPath' is not valid json. Error: " . $error);
        }
        // DOTO: validate manifest structure

        $this->manifests[$instance] = $manifest;

        return $manifest;
    }
}
