<?php

namespace IZON\Tasks\Services;

use IZON\Logs\Logger;
use IZON\Tasks\Domain\RegisteredTask;
use IZON\Tasks\Domain\TaskRun;
use IZON\Tasks\Domain\TaskState;
use IZON\Tasks\Exceptions\TaskException;
use IZON\Utils\Date;

/**
 * Description of TaskWatchdogService
 *
 * @author Lukáš Linhart <linhart@izon.cz>
 */
class TaskWatchdogService implements ITaskWatchdogService {
    /**
     *
     * @var TaskRunDao 
     */
    protected $taskRunDao;
    /**
     *
     * @var RegisteredTaskDao
     */
    protected $registeredTaskDao;
    /**
     *
     * @var RegisteredTaskTranslationsDao
     */
    protected $registeredTaskTranslationDao;
    /**
     *
     * @var Logger
     */
    protected $logger;

    public function __construct($loggerName, TaskRunDao $taskRunDao, RegisteredTaskDao $registeredTaskDao, RegisteredTaskTranslationsDao $registeredTaskTranslationDao) {
        $this->taskRunDao = $taskRunDao;
        $this->registeredTaskDao = $registeredTaskDao;
        $this->registeredTaskTranslationDao = $registeredTaskTranslationDao;
        if(empty($loggerName)) {
            $this->logger = Logger::getLogger(self::class);
        } else {
            $this->logger = Logger::getLogger($loggerName);
        }
    }

    /**
     * 
     * @param RegisteredTask $task
     * @return TaskRun
     */
    public function beginTaskRun(RegisteredTask $task) {
        $taskRun = new TaskRun();
        $taskRun->setFkRegisteredTaskId($task->getId());
        $taskRun->setFkTaskStateId(TaskState::TASK_STATE_RUNNING_ID);
        $taskRun->setStarted(new Date());
        $taskRun->setRegisteredTask($task);
        $this->taskRunDao->save($taskRun);
        return $taskRun;
    }

    public function endTaskRun(TaskRun $taskRun) {
        $taskRun->setEnded(new Date());
        $taskRun->setFkTaskStateId(TaskState::TASK_STATE_ENDS_ID);
        $this->taskRunDao->update($taskRun);
    }

    public function checkRunningTasks() {
        /* @var $tasks TaskRun[] */
        $tasks = $this->getRunnigTasks();
        $now = new Date();
        foreach($tasks as $taskRun) {
            $maxEndTime = new Date($taskRun->getStarted()->getTimestamp() + $taskRun->getRegisteredTask()->getMaxTime());
            if($maxEndTime->before($now)) {
                $message = 'Task ' . $taskRun->getRegisteredTask()->getTranslations()[0]->getName() . ' nebyl ukončen ve stanoveném čase.';
                $this->logger->warning($this->generateBody($message, $taskRun));
                $this->taskRunDao->update($taskRun);
            }
        }
    }

    protected function getRunnigTasks() {
        /* @var $tasks TaskRun[] */
        $tasks = $this->taskRunDao->find(['fkTaskStateId' => TaskState::TASK_STATE_RUNNING_ID])->listResult();
        foreach($tasks as $task) {
            $registeredTask = $this->registeredTaskDao->load($task->getFkRegisteredTaskId());
            $translations = $this->registeredTaskTranslationDao->find(['fkRegisteredTaskId' => $registeredTask->getId()])->listResult();
            $registeredTask->setTranslations($translations);
            $task->setRegisteredTask($registeredTask);
        }
        return $tasks;
    }

    /**
     * 
     * @param string $className
     * @return RegisteredTask
     */
    public function getRegisteredTask($className) {
        /* @var $registeredTask RegisteredTask */

        $registeredTask = $this->registeredTaskDao->find(['class' => $className])->uniqueResult();
        if(empty($registeredTask)) {
            throw new TaskException('No RegisteredTask found for class ' . $className . '.');
        }
        $translations = $this->registeredTaskTranslationDao->find(['fkRegisteredTaskId' => $registeredTask->getId()])->listResult();
        if(empty($translations)) {
            throw new TaskException('No RegisteredTaskTranslation found for class ' . $className . ' and RegisteredTask id = ' . $registeredTask->getId() . '.');
        }
        $registeredTask->setTranslations($translations);
        return $registeredTask;
    }

    protected function generateBody($message, TaskRun $taskRun) {
        $body = $message . "\n";
        $body .= print_r($taskRun, true);
        return $body;
    }

}
