<?php

namespace IZON\Tasks\BaseTasks;

use GoPay\Http\Log\Logger;
use IZON\MVC\Config;
use IZON\Tasks\Exceptions\TaskInitException;
use IZON\Tasks\Task;

/**
 * Created by PhpStorm.
 * User: honza
 * Date: 25.8.17
 * Time: 8:10
 */
abstract class LoggerTask implements Task {

    /**
     * Defautl path to directory where task will make logs
     * @var string
     */
    protected $logLocation = __BASE_DIR__ . "/tasks/logs";

    /**
     * Default name of file to save logs
     * @var string
     */
    protected $logName = 'taskLog.log';

    /**
     * String to write into log file
     * @var string
     */
    protected $toWrite = '';
    /**
     * Max size of one log file
     * In megabytes
     * @var int
     */
    protected $maxLogFileSize = 5;
    /**
     * Max count of backoup files of log
     * @var int
     */
    protected $maxBackupLogFiles = 2;
    /**
     * @var \IZON\Logs\Logger
     */
    protected $logger = null;
    /**
     * Determine if you want to send emails in log fatal or error
     * @var bool
     */
    protected $sendEmails = true;

    /**
     * Set location of log files
     *
     * Check if directory exist and if is writeable
     *
     * @param string $dirPath absolute path of directory
     * @throws TaskInitException
     */
    public function setLogLocation($dirPath) {
        if (!file_exists($dirPath)) {
            throw new TaskInitException('Directory \'' . $dirPath . '\' is not exist');
        }
        if (!is_writable($dirPath)) {
            throw new TaskInitException('Task can not write into \'' . $dirPath . '\' - change permissions');
        }
        $this->logLocation = $dirPath;
    }

    /**
     * Set log file name
     *
     * @param $fileName
     * @throws TaskInitException
     */
    public function setLogName($fileName) {
        $filePath = $this->logLocation . '/' . $fileName;

        if (file_exists($filePath)) {
            if (!is_writable($filePath)) {
                throw new TaskInitException('Task can not write into \'' . $fileName . '\' - change permissions');
            }
        } else {
            if (!is_writable($this->logLocation)) {
                throw new TaskInitException('Task can not write into \'' . $this->logLocation . '\' - change permissions');
            }
        }

        $this->logName = $fileName;
    }

    /**
     * Set max log file size
     * @param int $fileSize Max. file size of log in MB
     * @throws TaskInitException
     */
    public function setMaxLogFileSize($fileSize) {
        if (!is_int($fileSize)) {
            throw new TaskInitException('Max file size is not integer');
        }
        $this->maxLogFileSize = $fileSize;
    }

    /**
     * @param int $count count of max backup files of logger
     * @throws TaskInitException
     */
    public function setMaxBackupLogFiles($count) {
        if (!is_int($count)) {
            throw new TaskInitException('Max backoup files is not integer');
        }
        $this->maxBackupLogFiles = $count;
    }

    /**
     * True -> send error mails
     * False -> do not send emails
     * @param bool $bool
     */
    public function setSendEmails($bool){
        $this->sendEmails = $bool;
    }

    /**
     * Init logger
     * @throws TaskInitException
     */
    protected function initLogger() {
        $defConfig = Config::getValue('logs.config');
        if (empty($defConfig)) {
            throw new TaskInitException('Not found default logger config in config');
        }
        if(array_key_exists('rootLogger', $defConfig)) {
            $this->configureLogger($defConfig);
        } else {
            $this->configureMonologLogger($defConfig);
        }
    }
    
    protected function configureLogger($defConfig) {
        if (empty($defConfig['appenders']['default']['params'])) {
            throw new TaskInitException('Default config of logger faild');
        }
        $defConfig['appenders']['default']['params']['file'] = $this->logLocation . '/' . $this->logName;
        $defConfig['appenders']['default']['params']['maxFileSize'] = $this->maxLogFileSize . 'MB';
        $defConfig['appenders']['default']['params']['maxBackupIndex'] = $this->maxBackupLogFiles;
        if (!$this->sendEmails) {
            $defConfig['rootLogger']['appenders'] = ['default'];
        } else {
            if (empty($defConfig['appenders']['email'])) {
                throw new TaskInitException('Email config of logger is not exist');
            }
            $defConfig['appenders']['email']['params']['subject'] = $defConfig['appenders']['email']['params']['subject'] . ' - task';
        }
        \IZON\Logs\Logger::configure($defConfig);
        $this->logger = \IZON\Logs\Logger::getLogger(self::class);
    }
    
    protected function configureMonologLogger($defConfig) {      
        $level = \Psr\Log\LogLevel::WARNING;
        
        if(!array_key_exists('handlers', $defConfig)) {
            $defConfig['handlers'] = [];
        } else {
            if(array_key_exists('file', $defConfig['handlers'])) {
                $level = $defConfig['handlers']['file']->getLevel();
            }
        }
        
        
        if (!$this->sendEmails) {
            // NOT IMPLEMENTED
        } else {
            // NOT IMPLEMENTED
        }
        
        $fileSizeHadler = new \IZON\Logs\Handlers\RotatingFileSizeHandler(
            $this->logLocation . '/' . $this->logName,
            $this->maxBackupLogFiles + 1, 
            $this->maxLogFileSize . 'MB',
            $level
        );
        $defConfig['handlers']['file'] = $fileSizeHadler;
        
        \IZON\Logs\LoggerFactory::configure($defConfig);
        $this->logger = \IZON\Logs\Logger::getLogger(self::class);
    }

    /**
     * Write string into log like info
     * @var string $toWrite string to write
     */
    public function textToLog($toWrite) {
        $this->logger->info($toWrite);
    }

    /**
     * Write error into log
     * If send emails is enable - send email
     * @param $message
     * @param \Exception|null $exception
     */
    public function error($message, $exception = null) {
        $context = [];
        if ($exception != null) {
            $context['exception'] = $exception;
        }
        $this->logger->error($message, $context);
    }

    /**
     * Write fatal into log
     * If send emails is enable - send email
     * @param $message
     * @param null $exception
     */
    public function fatal($message, $exception = null) {
        $context = [];
        if ($exception != null) {
            $context['exception'] = $exception;
        }
        $this->logger->critical($message, $context);
    }

    /**
     * Init and run Task
     * On succes - write into log file 'success' string
     * Do not catch any exception
     * @param array $args
     */
    public function run(array $args) {
        $this->initLogger();
        $this->runTask();
        $this->textToLog('success');
    }

    abstract public function runTask();


}