<?php

namespace IZON\Tasks\Services;

use IZON\Mailer\Mail;
use IZON\Mailer\Mailer;
use IZON\MVC\Config;
use IZON\Tasks\Dao\RegisteredTaskDao;
use IZON\Tasks\Dao\RegisteredTaskTranslationsDao;
use IZON\Tasks\Dao\TaskRunDao;
use IZON\Tasks\Domain\RegisteredTask;
use IZON\Tasks\Domain\TaskRun;
use IZON\Tasks\Domain\TaskState;
use IZON\Tasks\Exceptions\TaskException;
use IZON\Utils\Date;

/**
 * Description of TaskWatchdogService
 *
 * @author Lukáš Linhart <linhart@izon.cz>
 */
class TaskWatchdogService implements ITaskWatchdogService {

  /**
   *
   * @var TaskRunDao 
   */
  protected $taskRunDao;

  /**
   *
   * @var RegisteredTaskDao
   */
  protected $registeredTaskDao;

  /**
   *
   * @var RegisteredTaskTranslationsDao
   */
  protected $registeredTaskTranslationDao;

  /**
   *
   * @var Mailer
   */
  protected $mailer;
  
  protected $sender;
  
  protected $emails;

  function __construct(Mailer $mailer, TaskRunDao $taskRunDao, RegisteredTaskDao $registeredTaskDao, RegisteredTaskTranslationsDao $registeredTaskTranslationDao) {
    $this->taskRunDao = $taskRunDao;
    $this->registeredTaskDao = $registeredTaskDao;
    $this->registeredTaskTranslationDao = $registeredTaskTranslationDao;
    $this->mailer = $mailer;
    $config = Config::getValue('dbTasks');
    if(empty($config)) {
      throw new TaskException('No configuration for dbTask');
    }
    if(empty($config['sender'])) {
      throw new TaskException('No configuration for dbTask sender');
    }
    
    if(empty($config['emails'])) {
      throw new TaskException('No configuration for dbTask emails');
    }
    $this->sender = $config['sender'];
    $this->emails = $config['emails'];
  }

    /**
   * 
   * @param RegisteredTask $task
   * @return TaskRun
   */
  public function beginTaskRun(RegisteredTask $task) {
    $taskRun = new TaskRun();
    $taskRun->setFkRegisteredTaskId($task->getId());
    $taskRun->setFkTaskStateId(TaskState::TASK_STATE_RUNNING_ID);
    $taskRun->setStarted(new Date());
    $taskRun->setRegisteredTask($task);
    $this->taskRunDao->save($taskRun);
    return $taskRun;
  }

  public function endTaskRun(TaskRun $taskRun) {
    $taskRun->setEnded(new Date());
    $taskRun->setFkTaskStateId(TaskState::TASK_STATE_ENDS_ID);
    $this->taskRunDao->update($taskRun);
  }

  public function checkRunningTasks() {
    /* @var $tasks TaskRun[] */
    $tasks = $this->getRunnigTasks();
    $now = new Date();
    foreach ($tasks as $taskRun) {
      $maxEndTime = new Date($taskRun->getStarted()->getTimestamp() + $taskRun->getRegisteredTask()->getMaxTime());
      if ($maxEndTime->before($now)) {
        $mail = $this->generateMail();
        if ($mail !== FALSE) {
          $mail->setSubject('Task ' . $taskRun->getRegisteredTask()->getTranslations()[0]->getName() . ' nebyl ukončen ve stanoveném čase.');
          $message = 'Task ' . $taskRun->getRegisteredTask()->getTranslations()[0]->getName() . ' nebyl ukončen ve stanoveném čase.';
          $mail->setBody($this->generateBody($message, $taskRun));
          $this->mailer->sendMail($mail);
        }
        $taskRun->setFkTaskStateId(TaskState::TASK_STATE_REPORTED_ID);
        $this->taskRunDao->update($taskRun);
      }
    }
  }
  
  protected function getRunnigTasks() {
    /* @var $tasks TaskRun[] */
     $tasks = $this->taskRunDao->find(['fkTaskStateId' => TaskState::TASK_STATE_RUNNING_ID])->listResult();
     foreach ($tasks as $task) {
       $registeredTask = $this->registeredTaskDao->load($task->getFkRegisteredTaskId());
       $translations = $this->registeredTaskTranslationDao->find(['fkRegisteredTaskId' => $registeredTask->getId()])->listResult();
       $registeredTask->setTranslations($translations);
       $task->setRegisteredTask($registeredTask);
     }
     return $tasks;
  }

  /**
   * 
   * @param string $className
   * @return RegisteredTask
   */
  public function getRegisteredTask($className) {
    /* @var $registeredTask RegisteredTask */
    
    $registeredTask = $this->registeredTaskDao->find(['class' => $className])->uniqueResult();
    if(empty($registeredTask)) {
      throw new TaskException('No RegisteredTask found for class '.$className.'.');
    }
    $translations = $this->registeredTaskTranslationDao->find(['fkRegisteredTaskId' => $registeredTask->getId()])->listResult();
    if(empty($translations)) {
      throw new TaskException('No RegisteredTaskTranslation found for class '.$className.' and RegisteredTask id = '.$registeredTask->getId().'.');
    }
    $registeredTask->setTranslations($translations);
    return $registeredTask;
  }

  protected function generateBody($message, TaskRun $taskRun) {
    $body = $message . "\n";
    $body .= print_r($taskRun, true);
    return $body;
  }

  /**
   * 
   * @return boolean|Mail
   */
  protected function generateMail() {
    if (empty($this->emails) || empty($this->sender)) {
      return FALSE;
    }
    $mail = new Mail();
    $mail->setFrom($this->sender);
    foreach ($this->emails as $email) {
      $mail->addTo($email);
    }
    return $mail;
  }

}
