<?php

namespace IZON\PaymentGate\Services;


use IZON\PaymentGate\Dao\PaymentGateDao;
use IZON\PaymentGate\Dao\PaymentGateTransactionDao;
use IZON\PaymentGate\Dao\PaymentGateTransactionProcessDao;
use IZON\PaymentGate\Domain\PaymentGate;
use IZON\PaymentGate\Domain\PaymentGateTransaction;
use PHPMailer\PHPMailer\Exception;

abstract class BasePaymentGateService{

    /**
     * @var PaymentGateDao
     */
    protected $paymentGateDao;
    /**
     * @var PaymentGateTransactionDao
     */
    protected $paymentGateTransactionDao;
    /**
     * @var PaymentGateTransactionProcessDao
     */
    protected $paymentGateTransactionProcessDao;

    /**
     * BasePaymentGateService constructor.
     * @param PaymentGateDao $paymentGateDao
     * @param PaymentGateTransactionDao $paymentGateTransactionDao
     * @param PaymentGateTransactionProcessDao $paymentGateTransactionProcessDao
     */
    public function __construct(PaymentGateDao $paymentGateDao, PaymentGateTransactionDao $paymentGateTransactionDao, PaymentGateTransactionProcessDao $paymentGateTransactionProcessDao) {
        $this->paymentGateDao = $paymentGateDao;
        $this->paymentGateTransactionDao = $paymentGateTransactionDao;
        $this->paymentGateTransactionProcessDao = $paymentGateTransactionProcessDao;
    }


    /**
     * @param $id
     * @return PaymentGate
     */
    public function getPaymentGate($id){
        /** @var PaymentGateTransaction $gate */
        $gate = $this->paymentGateDao->load($id);
        return $gate;
    }

    /**
     * @param PaymentGateTransaction $transaction
     */
    protected function fillTransaction(PaymentGateTransaction $transaction){
        $process = $this->paymentGateTransactionProcessDao->find(['fkPaymentTransactionId' => $transaction->getId()])->listResult();
        $transaction->setProcess($process);
    }

    /**
     * @param $name string of name
     * @return PaymentGate | null
     */
    public function getPaymentGateByName($name){
        try {
            $transactions = $this->paymentGateTransactionDao->find(["name" => $name])->uniqueResult();
        }catch(\Exception $e){
            return null;
        }
        return $transactions;
    }

    /**
     * Save into db
     * @param PaymentGateTransaction $transaction
     */
    public function saveTransaction(PaymentGateTransaction $transaction){
        $this->paymentGateTransactionDao->beginTransaction();
        $this->paymentGateDao->save($transaction);
        $this->updateProcessObjects($transaction);
        $this->paymentGateTransactionDao->commit();
    }

    /**
     * Update in d
     * @param PaymentGateTransaction $transaction
     */
    public function updateTransaction(PaymentGateTransaction $transaction){
        $this->paymentGateTransactionDao->beginTransaction();
        $this->paymentGateDao->update($transaction);
        $this->updateProcessObjects($transaction);
        $this->paymentGateTransactionDao->commit();
    }

    /**
     * List of transactions
     * @param $id int
     * @return PaymentGateTransaction[]
     */
    public function getTransactionsByPayGate(PaymentGate $payGate){
        $transactions = $this->paymentGateTransactionDao->find(["fkPaymentGateId" => $payGate->getId()])->listResult();
        foreach($transactions as $transaction){
            $this->fillTransaction($transaction);
        }
        return $transactions;
    }


    /**
     * List of Transactions
     * @param $code string code of transaction
     * @return PaymentGateTransaction|null
     * @throws
     */
    public function getTransactionByCode($code){
        /** @var PaymentGateTransaction $transaction */
        $transaction = $this->paymentGateTransactionDao->find(["code" => $code])->uniqueResult();

        if ($transaction instanceof PaymentGateTransaction){
            $this->fillTransaction($transaction);
        }

        return $transaction;
    }


    /**
     * To update paymentTransactionProcess
     * @param PaymentGateTransaction $transaction
     */
    public function updateProcessObjects(PaymentGateTransaction $transaction){
        $olds = $this->paymentGateTransactionProcessDao->find(['fkPaymentTransactionId' => $transaction->getId()])->listResult();
        $oldsHash = [];
        foreach($olds as $old){
            $oldsHash[$old->getId()] = $old;
        }
        foreach($transaction->getProcess() as $object){
            if(isset($oldsHash[$object->getId()])){
                unset($oldsHash[$object->getId()]);
            }else{
                $object->setFkPaymentTransactionId($transaction->getId());
                $this->paymentGateTransactionProcessDao->save($object);
            }
        }
    }

}
