<?php

namespace IZON\Composer\Plugins\ProjectsInfo;

use Composer\Composer;
use Composer\EventDispatcher\EventSubscriberInterface;
use Composer\IO\IOInterface;
use Composer\Plugin\PluginInterface;
use Composer\Script\Event;
use Composer\Package\PackageInterface;
use function IZON\String\endsWith;
use function IZON\String\startsWith;

/**
 * Sends project info on composer update.
 */
class ProjectsInfo implements PluginInterface, EventSubscriberInterface {

    /**
     * @var Composer 
     */
    protected $composer;

    /**
     *
     * @var IOInterface 
     */
    protected $io;
    
    /**
     * URL of API for sending data.
     * @var string
     */
    protected $apiUrl = 'https://pi.izon.cz/api';

    /**
     * PluginInterface method to recieve composer dependencies.
     * @param Composer $composer
     * @param IOInterface $io
     */
    public function activate(Composer $composer, IOInterface $io) {
        $this->composer = $composer;
        $this->io = $io;
    }

    /**
     * Is called by Composer to register listener methods on specific events.
     * @return array
     */
    public static function getSubscribedEvents() {
        return [
            "post-install-cmd" => [
                ['logProjectInfo', 0],
            ],
            "post-update-cmd" => [
                ['logProjectInfo', 0],
            ]
        ];
    }

    /**
     * Logs project info.
     * @param Event $event
     */
    public function logProjectInfo(Event $event) {
        
        # Check if project is setup correctly to be able to send all needed data
        if(!$this->checkProjectSettings()) {
            return;
        }
        
        # Log only on production
        if(!$this->isProduction()) {
            return;
        }
        
        # Create project data array (for JSON)
        $projectData = $this->createProjectData();
        $json = json_encode($projectData);

        # Send JSON
        // DOTO implement some authentication headers for API auth
        $ch = curl_init($this->apiUrl.'/projects');                                                                      
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "PUT");                                                                     
        curl_setopt($ch, CURLOPT_POSTFIELDS, $json);                                                                  
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);                                                                      
        curl_setopt($ch, CURLOPT_HTTPHEADER, array(                                                                          
            'Content-Type: application/json',                                                                                
            'Content-Length: ' . strlen($json))                                                                       
        );                                                                                                                   

        $result = curl_exec($ch);
    }
    
    /**
     * Checks if projects settings is correct.
     * @return bool
     */
    protected function checkProjectSettings() {
        # Must have project name in composer.json
        if(empty($this->composer->getPackage()->getName())) {
            return false;
        }
        return true;
    }

    /**
     * Gets project data array.
     * @return array
     */
    protected function createProjectData() {
        return [
            'name' => $this->composer->getPackage()->getName(),
            'packages' => $this->getPackagesData()
        ];
    }

    /**
     * Gets package data array.
     * @return array
     */
    protected function getPackagesData() {
        $data = [];
        $packages = $this->composer
                ->getRepositoryManager()
                ->getLocalRepository()
                ->getPackages();
        foreach($packages as $package) {
            if($this->isDevPackage($package)) {
                continue;
            }
            $data[] = [
                'name' => $package->getName(),
                'version' => $package->getVersion()
            ];
        }
        return $data;
    }

    /**
     * Checks if package is for dev.
     * @param PackageInterface $package
     * @return boolean
     */
    protected function isDevPackage(PackageInterface $package) {
        $devString = 'dev';
        if(endsWith($package->getVersion(), $devString)) {
            return true;
        }
        if(startsWith($package->getVersion(), $devString)) {
            return true;
        }
        return false;
    }
    
    /**
     * Checks if is run on production.
     * @return boolean
     */
    protected function isProduction() {
        if(empty(getenv('MACHINE_ID'))) {
            return true;
        }
        return false;
    }

    public function deactivate(Composer $composer, IOInterface $io) {
        // TODO: Implement deactivate() method.
    }

    public function uninstall(Composer $composer, IOInterface $io) {
        // TODO: Implement uninstall() method.
    }
}
