<?php

namespace IZON\MVC\Views;

use Exception;
use Throwable;
use IZON\Logs\Logger;
use IZON\MVC\Context\Context;
use function IZON\String\endsWith;
use function IZON\String\startsWith;


/**
 * trida renderujici view, resici cachovani rendrovani view, skladajici view z jednotlivych casti, ...
 * jednotliva view jsou ulozena v adresarich
 * pri devel modu naincluduje nekde dole ve strance misto na debug informace
 */
class PHPView implements View {

    const VIEW_SUFFIX = ".tmpl.php";

    /**
     * suffix of library files
     */
    const LIBRARY_SUFFIX = ".php";

    /**
     * view to render
     * @var View
     */
    private static $applicationView;

    /**
     * @var string name of view to render
     */
    protected $viewPath;

    /**
     * @var array parameters to use in template
     */
    protected $parameters;

    /**
     * @var bool
     */
    protected $useOutputBuffer = true;


    /**
     * @var Context|null
     */
    protected $context = null;

    /**
     * @var string[] dir containing libraries for use in view
     */
    protected $librariesDirs = [];

    /**
     * @var string[] files containing libraries used for rendering the view
     */
    protected $librariesFiles = [];

    /**
     * @var string[] directory where to find static content
     */
    protected $staticContentDirs = [];

    /**
     * @var string dir to put minified content
     */
    protected $minifiedStaticContentCachePrefix = null;

    /**
     * @var string|null
     */
    protected $minifiedStaticContentCacheDir = null;


    /**
     * @var string[] prefiexs to cicle through for statis files to balance load for fifferent servers
     */
    protected $staticContentPrefixes = [];

    /**
     * @var integer
     */
    protected $lastContentPrefixIndex = 0;

    /**
     * @var Logger
     */
    protected $log;

    /**
     * vytvari view podle zadaneho jmena a predava mu parametry
     */
    function __construct($viewPath) {
        $this->viewPath = $viewPath;

        $this->log = Logger::getLogger(self::class);
    }

    function getLocale() {
        return $this->context->getLocale();
    }

    function getParameters() {
    	return $this->parameters;
    }

    function getParameter($paramName) {
        if( isset($this->parameters[$paramName]) ) {
            return $this->parameters[$paramName];
        } else {
            return NULL;
        }
    }

    function hasParameter($paramName) {
        return array_key_exists($paramName, $this->parameters);
    }

    /**
     * vrati router
     */
    function getRouter() {
        return $this->context->getRouter();
    }

    /**
     * @return Context
     */
    function getContext(): Context {
        return $this->context;
    }


    /**
     * provede vykresleni celeho view
     */
    function render(bool $asString = false): ?string {
        if( $this->useOutputBuffer || $asString) {
            // TODO: handle output buffer better
            ob_start();  // start buffering output
        }

        try {
            // load view libraries from dirs
            foreach($this->librariesDirs as $libraryDir) {
                if( file_exists($libraryDir) && is_dir($libraryDir) ) { // test if lib dir exists
                    foreach(scandir($libraryDir) as $file) {
                        if( endsWith($file, self::LIBRARY_SUFFIX) ) { // load only php files
                            require_once $libraryDir ."/". $file;
                        }
                    }
                }
            }

            // require view to be shown
            require $this->viewPath;

            if( $this->useOutputBuffer && !$asString) {
                ob_end_flush(); // end buffering and flush buffer to output
            } else if($asString) {
                if($asString) {
                    $view = ob_get_contents();
                    ob_end_clean();
                    return $view;
                }
            }
        } catch(Throwable $e) {
            if( $this->useOutputBuffer ) {
                ob_end_clean();
            }
            throw $e;
        }
        return null;
    }

    /**
     * nacte staticke aplikacni view
     * @return View aplikacni viw
     */
    static function getApplicationView() {
    	return self::$applicationView;
    }

    /**
     * ulozi staticke aplikacni view
     */
    static function setApplicationView($applicationView) {
    	self::$applicationView = $applicationView;
    }

    /**
     * inits view with necessary data
     * @param Context $context
     * @param array $parameters
     */
    public function init(Context $context, array $parameters) {
        $this->parameters = $parameters;
        $this->context = $context;

        self::setApplicationView($this);
    }

    public function getStaticFileURL($urlPath) {
        foreach($this->staticContentDirs as $urlPrefix => $filePrefix) {
            if( startsWith($urlPath, $urlPrefix) ) {
                $strippedPath = mb_substr($urlPath, mb_strlen($urlPrefix));
                if( startsWith($strippedPath, "/") ) { // remove initial / to be reative path
                    $strippedPath = mb_substr($strippedPath, 1);
                }

                if($this->minifiedStaticContentCacheDir !== null ) { // minified files cache dir provided
                    // try to find minified file
                    $filePrefixDir = basename($filePrefix);
                    $filePath = $this->minifiedStaticContentCacheDir ."/". $filePrefixDir ."/". $strippedPath;
                    if( file_exists($filePath)
                        && !is_dir($filePath) )  { // required file exists
                        return $this->getNextStaticContentURLPrefix() . $this->minifiedStaticContentCachePrefix ."/". $filePrefixDir ."/". $strippedPath ."?". filemtime($filePath);
                    }
                }

                $filePath = $filePrefix ."/". $strippedPath;
                if( file_exists($filePath)
                    && !is_dir($filePath) )  {
                    return $this->getNextStaticContentURLPrefix() . $urlPath ."?ts=". filemtime($filePath);
                }

                throw new Exception("Static file with URL ". $urlPath ." doesn't exist in folder ". $filePrefix);
            }
        }

        throw new Exception("Non existing directory for file ". $urlPath);
    }

    protected function getNextStaticContentURLPrefix() {
        $this->lastContentPrefixIndex = ($this->lastContentPrefixIndex + 1) % count($this->staticContentPrefixes);
        $prefix = $this->staticContentPrefixes[$this->lastContentPrefixIndex];
        return $prefix;
    }


/// generated setters
    function setLibrariesDirs($librariesDirs) {
        $this->librariesDirs = $librariesDirs;
    }

    function setLibrariesFiles($librariesFiles) {
        $this->librariesFiles = $librariesFiles;
    }

    function setStaticContentDirs($staticContentDirs) {
        $this->staticContentDirs = $staticContentDirs;
    }

    function setStaticContentPrefixes($staticContentPrefixes) {
        $this->staticContentPrefixes = $staticContentPrefixes;
    }

    function setUseOutputBuffer($useOutputBuffer) {
        $this->useOutputBuffer = $useOutputBuffer;
    }

    function setMinifiedStaticContentCacheDir($minifiedStaticContentCacheDir) {
        $this->minifiedStaticContentCacheDir = $minifiedStaticContentCacheDir;
    }

    function setMinifiedStaticContentCachePrefix($minifiedStaticContentCachePrefix) {
        $this->minifiedStaticContentCachePrefix = $minifiedStaticContentCachePrefix;
    }
}
