<?php

namespace IZON\MVC\Routers;

use \Exception;

use IZON\MVC\URLDomain\URLDomainResolverInterface;
use \IZON\Utils\Locale;
use IZON\MVC\Locale\LocaleResolver;

use IZON\MVC\Messages\HttpRequestInterface;
use IZON\MVC\Routers\RouteInfo;
use \IZON\MVC\Exceptions\PageNotFoundException;


/**
 * finds controller findRoute for HttpRequest
 * finds url findURL for $controllerId
 * url => controller and controllerId => url translations are defined in $urlRouteDefinitions passed in constructor
 */
class Router {

    /**
     * @var RouteDefinition[] definitions of routes
     */
    protected $urlRouteDefinitions = [];

    /**
     * @var string[] interceptors to  be called for each request resolved by this router
     */
    protected $commonInterceptors = [];

    /**
     * @var LocaleResolver
     */
    protected $localeResolver;
    /**
     * @var URLDomainResolverInterface
     */
    protected $domainResolver;

    /**
     * @var boolean pokud je true tak pro stranky s koncovym lomitkem v url provede externi presmervani na stranku bez neho pokud takove stranky existuji
     */
    protected $redirectToURLWithoutTailingSlash = true;
    /**
     * @var string
     */
    protected $defaultProtocol;


    public function __construct(array $urlRouteDefinitions, string $defaultProtocol = 'http') {
        $this->urlRouteDefinitions = $urlRouteDefinitions;
        $this->defaultProtocol = $defaultProtocol;
    }

    /**
     * pro request najde informace, jaky controller se ma volat a jake interceptory
     * @param HttpRequestInterface $httpRequest
     * @return RouteInfo
     * @throws Exception
     */
    public function findRoute(HttpRequestInterface $httpRequest) {
        $url = $httpRequest->getURL();
        $domainUID = $this->domainResolver->resolveDomainUID($httpRequest->getDomain());
        // removes tailing /
        if( $this->redirectToURLWithoutTailingSlash
            && \IZON\String\endsWith($url, "/")
            &&  $url != '/') {
            $uri = $httpRequest->getUri(); // TODO: change uri
            $urlWithoutSlash = mb_substr($url, 0, mb_strlen($url)-1); // urizni koncove lomitko
            $newRequest = $httpRequest->withUri($uri);
            $newRequest->setURL($urlWithoutSlash);

            // try to find route for url without tailing /
            foreach( $this->urlRouteDefinitions as $definition ) {
                $routeInfo = $definition->findRoute($newRequest);
                if( $routeInfo != NULL ) {
                    // add locale data to url
                    $newUrl = $this->localeResolver->modifyOutpultURL($domainUID, $urlWithoutSlash, $newRequest->getLocale());
                    $routeInfo = new RouteInfo();
                    $routeInfo->setRedirectCode(true);
                    $routeInfo->setURL($newUrl);

                    // send redirect to new address
                    return $routeInfo;
                }
            }
        }

        // try to find address as is
        foreach( $this->urlRouteDefinitions as $definition ) {
            $routeInfo = $definition->findRoute($httpRequest);
            if( $routeInfo != NULL ) {
                // prida spolecne interceptory na zacatek pole
                $interceptors = $this->commonInterceptors;
                $interceptors = array_merge($interceptors, $routeInfo->getInterceptors());
                $routeInfo->setInterceptors($interceptors);

                // prepare context
                $context = $routeInfo->getContext();
                if( $context !== NULL ) {
                    $context->setLocale($httpRequest->getLocale());
                    $context->setRouter($this);
                }

                return $routeInfo;
            }
        }

        // no routing find
        throw new PageNotFoundException("Routing of address ". $httpRequest->getURL() ." with parameters: ". print_r($httpRequest->getParameters(), true) ." not found");
    }

    /**
     * pro controller s id $controllerId najde adresu pres kterou je ho mozno volat
     * @param string $controllerId id controlleru
     * @param array $parameters parametry, ptere se maji predat do adresy
     * @param string $methodName nazev metody, ktera se ma na controlleru zavolat
     * @param Locale $locale
     * @return string
     * @throws Exception
     */
    public function findURL($controllerId, array $parameters, $methodName, $locale, $domainUID = null, bool $forceDomain = false) {
        $currentDomainUID = $this->domainResolver->resolveCurrentDomain();
        if(empty($domainUID)) {
            $domainUID = $currentDomainUID;
        }
        foreach( $this->urlRouteDefinitions as $definition ) {
            $url = $definition->findURL($controllerId, $parameters, $methodName, $locale, $domainUID);
            if( $url != NULL ) {
                $url = $this->localeResolver->modifyOutpultURL($domainUID, $url, $locale);
                if($domainUID !== $currentDomainUID || $forceDomain) {
                    $url = $this->getProtocol().'://'.$this->domainResolver->resolveDomainName($domainUID).$url;
                }
                return $url;
            }
        }
        throw new Exception("URL for controller $controllerId and parameters ". print_r($parameters, true) ." not found");
    }

    function setCommonInterceptors(array $commonInterceptors) {
        $this->commonInterceptors = $commonInterceptors;
    }

    function setLocaleResolver(LocaleResolver $localeResolver) {
        $this->localeResolver = $localeResolver;
    }

    /**
     * @param URLDomainResolverInterface $domainResolver
     */
    public function setDomainResolver(URLDomainResolverInterface $domainResolver): void {
        $this->domainResolver = $domainResolver;
    }

    function setRedirectToURLWithoutTailingSlash($redirectToURLWithoutTailingSlash) {
        $this->redirectToURLWithoutTailingSlash = $redirectToURLWithoutTailingSlash;
    }

    protected function getProtocol() {
        if(
            (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off')
            || (isset($_SERVER['SERVER_PORT']) &&$_SERVER['SERVER_PORT'] == 443)
            || (isset($_SERVER['HTTP_X_FORWARDED_PROTO']) && $_SERVER['HTTP_X_FORWARDED_PROTO'] == 'https')
        ) {
            return 'https';
        }
        return  $this->defaultProtocol;
    }
}
