<?php

namespace IZON\MVC\Locale;

use Exception;

use IZON\MVC\Gettext\GettextLocaleManager;
use IZON\Utils\Locale;
use function IZON\String\startsWith;


/**
 *  Locale resolver that takes locale from start of url
 */
class URLLocaleResolver implements LocaleResolver {
    /**
     *
     * @var string defaultni locale, ktere se ma pouzit pokud neni zjisteno jine
     */
    protected $localeConfig;
    /**
     * @var GettextLocaleManager
     */
    protected $gettextManager;


    /**
     * URLLocaleResolver constructor.
     * @param array $localeConfig
     * @param GettextLocaleManager $gettextManager
     * @throws Exception
     */
    public function __construct(array $localeConfig, GettextLocaleManager $gettextManager) {
        $config = [];
        foreach($localeConfig as $domain => $domainConfig) {

            $lcConfig = [];
            foreach($domainConfig['supported'] as $key => $localeTag) {
                $lcConfig[$key] = Locale::forLocaleTag($localeTag);
            }
            $domainConfig['supported'] = $lcConfig;
            $config[$domain] = $domainConfig;
        }
        $this->localeConfig  = $config;
        $this->gettextManager = $gettextManager;
    }

    /**
     * @param string $domain
     * @param string $requestURL
     * @param array $requestParams
     * @return Locale
     * @throws Exception
     */
    public function resolveLocale(string $domain, string $requestURL, array $requestParams) {
        $domainConfig = $this->getDomainConfig($domain);

        $supported = $domainConfig['supported'];
        $default = $domainConfig['default'];
        $selectedLocale = $supported[$default];

        foreach($supported as $key => $locale) {
            if($key === $default) {
                continue;
            }
            if($requestURL !== '/'. $key && !startsWith($requestURL, '/'. $key .'/')) {
                continue;
            }
            $selectedLocale = $locale;
            break;

        }
        $this->gettextManager->setupLanguage($selectedLocale);

        return $selectedLocale;
    }

    /**
     * @param string $domain
     * @param string $requestURL
     * @param Locale $locale
     * @return false|string
     * @throws Exception
     */
    public function modifyRequestURL(string $domain, string $requestURL, Locale $locale) {
        $domainConfig = $this->getDomainConfig($domain);

        $supported = $domainConfig['supported'];
        $default = $domainConfig['default'];

        // odstran language z url
        $localeName = array_search($locale, $supported);
        if( $localeName === false ) {
            throw new Exception("Unsupported locale ". $locale->toLocaleTag());
        }

        if( $localeName ===  $default) { // is default locale do not modify
            return $requestURL;
        }

        if($requestURL == "/" . $localeName
            || startsWith($requestURL, "/" . $localeName . "/")) { // uriznout jen pokud je samotne nebo zacina a ma dalsi cast
            $ret = mb_substr($requestURL, mb_strlen("/" . $localeName));
            if($ret == '') {
                $ret = '/';
            }
            return $ret;
        } else {
            return $requestURL;
        }
    }

    /**
     * @param string $domain
     * @param string $requestURL
     * @param Locale $locale
     * @return string
     * @throws Exception
     */
    public function modifyOutpultURL(string $domain, string $requestURL, Locale $locale) {
        $domainConfig = $this->getDomainConfig($domain);

        $supported = $domainConfig['supported'];
        $default = $domainConfig['default'];

        // odstran language z url
        $localeName = array_search($locale, $supported);
        if( $localeName === false ) {
            throw new Exception("Unsupported locale ". $locale->toLocaleTag());
        }

        if( $localeName === $default ) { // is default locale do not modify
            return $requestURL;
        }

        return "/". $localeName . $requestURL;
    }

    /**
     * @param $domain
     * @return array
     * @throws Exception
     */
    protected function getDomainConfig($domain) {
        if(!array_key_exists($domain,$this->localeConfig)) {
            throw new \Exception("Unsupported domain ". $domain);
        }
        return $this->localeConfig[$domain];
    }

}
