<?php

namespace IZON\MVC\Routers;

use Exception;
use IZON\MVC\Exceptions\PageNotFoundException;
use IZON\Utils\Locale;
use IZON\MVC\Messages\HttpRequestInterface;
use IZON\MVC\Context\AjaxContext;
use Psr\Container\ContainerInterface;

/**
 * refinuje routovani pro ajaxive dotazy
 */
class AjaxRouteDefinition implements RouteDefinition {

    /**
     * interceptory pridavane pred vsechny controllery teto oute definition
     * @var array
     */
    protected $interceptors = [];

    /**
     * @var array array of arrays. Key is domainUID value is array of interceptor identifiers
     */
    protected $domainInterceptors = [];

    /**
     * interceptory vazane na jednotlive controllery
     * pole poli, klic je uid controlleru
     * a hodnota je pole s id jednotlivych controlleru
     * @var array
     */
    protected $controllerInterceptors = [];

    /**
     * @var string jak maji zacinat funkce, ktere se pouzivaji pro ajax
     */
    protected $methodPrefix = "ajax";

    /** @var ContainerInterface|null */
    protected $container = null;
    
    public function findRoute(HttpRequestInterface $request) {
        if( !\IZON\String\startsWith($request->getURL(), "/_ajax/") ) { // neobsluhuje se timto routerem
            return NULL;
        }

        if( preg_match('#^/_ajax/([A-Za-z0-9.]+)/([A-Za-z0-9]+)$#', $request->getURL(), $matches) ) {  // reqexp musi byt obalen delimiterem
            $routeInfo = new RouteInfo();
            $routeInfo->setURL($request->getURL());
            $routeInfo->setContrlollerId($matches[1]);
            $routeInfo->setMethodName($matches[2]);
            $routeInfo->setParameters($request->getParameters());

            if($this->container !== null) {
                if(!$this->container->has($matches[1])) {
                    throw new PageNotFoundException("Url ". $request->getURL() ." doesn`t exists");
                }
            }

            $interceptors = $this->interceptors;
            // add domain interceptors
            if( array_key_exists($request->getDomainUID(), $this->domainInterceptors) ) {
                $interceptors = array_merge($interceptors, $this->domainInterceptors[$request->getDomainUID()]);
            }
            if( isset($this->controllerInterceptors[$matches[1]]) ) {
                $interceptors = array_merge($interceptors, $this->controllerInterceptors[$matches[1]]);
            }
            $routeInfo->setInterceptors($interceptors);

            $context = new AjaxContext($request->getSession());
            $routeInfo->setContext($context);

            // nastavuje id volaneho controlleru
            $request->setCalledControllerId($matches[1]);

            return $routeInfo;
        } else {
            throw new PageNotFoundException("Url ". $request->getURL() ." must be in format /_ajax/controllerId/methodName");
        }
    }

    public function findURL($controllerId, array $parameters, $methodName, Locale $locale, ?string $domainUID) {
        if(\IZON\String\startsWith($methodName, $this->methodPrefix) ) { // je metoda pro obskuhu ajaxu
            $outputURL = "/_ajax/". $controllerId ."/". $methodName;

            $appendArray = [];
            foreach($parameters as $index => $value) {
                $appendArray[] = $index ."=". $value;
            }
            if( !\IZON\Arrays\isEmpty($appendArray) ) {
                $outputURL .= "?". implode('&', $appendArray);
            }
            return $outputURL;

        }
        return NULL;
    }

    /**
     * nastavuje interceptory platne pro vsechny controllery z teto route definition
     * @param array $interceptors
     */
    function setInterceptors(array $interceptors) {
        $this->interceptors = $interceptors;
    }

    /**
     * sets interceptors only applied for $domainUID domain
     * @param string $domainUID uid of domain interceptors are applied to
     * @param string[] $interceptors
     */
    function setDomainInterceptors(string $domainUID, array $interceptors) {
        $this->domainInterceptors[$domainUID] = $interceptors;
    }

    /**
     * @param ContainerInterface $container
     */
    function setContainer(ContainerInterface $container) {
        $this->container = $container;
    }
    
    /**
     * nastavuje interceptory jen pro specificke controllery
     * @param array $interceptors pole poli kde klicem je identifier controlleru a hodnota je pole interceptoru, ktere se na nem maji provezt
     */
    function setControllerInterceptors(array $interceptors) {
        $this->controllerInterceptors = $interceptors;
    }

    /**
     * adds ides of interceptors to be called around $controllerId
     * @param string $controllerId
     * @param string[] $interceptorIdes
     * @throws Exception
     */
    function addControllerInterceptors($controllerId, array $interceptorIdes) {
        if( array_key_exists($controllerId, $this->controllerInterceptors) ) {
            throw new Exception("Interceptors for controller $controllerId already added.");
        }
        $this->controllerInterceptors[$controllerId] = $interceptorIdes;
    }
}
