<?php

namespace IZON\MVC\Routers;

use Exception;
use IZON\DB\ConnectionInterface;
use IZON\Logs\Logger;
use IZON\MVC\Context\WebContext;
use IZON\MVC\Messages\HttpRequestInterface;
use IZON\MVC\Interceptors\PageInfoInterceptor;
use IZON\MVC\PageInfo;
use IZON\Utils\Locale;
use PDO;
use PDOException;
use function IZON\Arrays\isEmpty;
use function IZON\String\startsWith;

/**
 * novejsi definice routovani prodle db tabulek core_mvc_structure_*
 * TODO: should be refactored
 */
class DBRouteDefinition implements RouteDefinition {
    
    /**
     * @var ConnectionInterface
     */
    protected $dbConnection;
    
    /**
     * interceptory pridavane pred vsechny controllery teto oute definition
     * @var array 
     */
    protected $interceptors = [];

    /**
     * interceptory vazane na jednotlive controllery
     * pole poli, klic je uid controlleru 
     * a hodnota je pole s id jednotlivych controlleru
     * @var array 
     */
    protected $controllerInterceptors = [];

    /**
     *
     * @var string jaky maji mit prefix tabulky se strukturou webu
     */
    protected $structureTablePreffix = "core_mvc_structure_";
    
    /**
     *
     * @var Logger 
     */
    protected $log;
    
    
    public function __construct(ConnectionInterface $dbConnection) {
        $this->dbConnection = $dbConnection;
        
        // pridava PageInfoInterceptor ktery nacit informace o prave zobrazovane strance
        $this->interceptors[] = new PageInfoInterceptor();
        
        $this->log = Logger::getLogger(self::class);
    }
    
    public function findRoute(HttpRequestInterface $request) {
        $conn = $this->dbConnection;
        
        // kd pro kterou zemi se ma brat
        $countryCode = mb_strtolower($request->getLocale()->getCountry());
        
        // TODO: mohlo by se prekladat na 
        $sql = 'SELECT * FROM '. $this->structureTablePreffix . $countryCode .' WHERE enabled = 1 order by routes_order asc';
        try {
            $statement = $conn->prepare($sql);
            $statement->execute();
        } catch(PDOException $ex) {
            $message = "Při routovani adresy ". $request->getURL() ." došlo k chybě.";
            $this->log->error($message, ["exception" => $ex]);
            throw new Exception($message, 0, $ex);
        }
        
        while( $row = $statement->fetch(PDO::FETCH_ASSOC) ) {
            $urlPattern = $row['url'];

            $matches = [];
            //TODO - maybe use urldecode to ($request->getURL) - Honza
            if( preg_match('#'. $urlPattern .'#', $request->getURL(), $matches) ) {  // reqexp musi byt obalen delimiterem
                $parameters = $matches;
                unset($parameters[0]);
                
                $newQueryParametes = [];
                // prekopirovat parametry z adresy do $request
                foreach($parameters as $key => $value) {
                    if( !is_numeric($key) ) { // numeric values represent index of match, we dont need them as we bind reqex values to names only
                        $newQueryParametes[$key] = $value;
                    }
                }
                
                $urlQueryParametes = [];
                // prekopirovat parametry z adresy do $request
                foreach($parameters as $key => $value) {
                    if( !is_numeric($key) ) { // numeric values represent index of match, we dont need them as we bind reqex values to names
//                        $request->addParameter($key, $value);
                        $newQueryParametes[$key] = $value;
                        $urlQueryParametes[$key] = $value;
                    }
                }
                
                // nastavi pageinfo do HttpRequest, aby bylo pristupne pro controller 
                $pageInfo = $this->createPageInfo($conn, $row, $request->getURL(), $countryCode);
                $pageInfo->setParameters($newQueryParametes);
                
                $routeInfo = new RouteInfo();
                $routeInfo->setURL($request->getURL());
                $routeInfo->setPattern($urlPattern);
                $routeInfo->setContrlollerId($row['controller_uid']);
                if( $row['method_name'] ) { // ma se posilat na jinou nez defaultni metodu
                    $routeInfo->setMethodName($row['method_name']);
                }
                $routeInfo->setParameters($urlQueryParametes);
                
                // add context to page
                $context = new WebContext($request->getSession(), $pageInfo);
                $routeInfo->setContext($context);
                
                $interceptors = $this->interceptors;
                if( isset($this->controllerInterceptors[$row['controller_uid']]) ) {
                    $interceptors = array_merge($interceptors, $this->controllerInterceptors[$row['controller_uid']]);
                }
                
                $routeInfo->setInterceptors($interceptors);
                
                // nastavuje id volaneho controlleru
                $request->setCalledControllerId($row['controller_uid']);
                
                return $routeInfo;
            }
        }
        return NULL;
    }

    public function findURL($controllerId, array $parameters, $methodName, Locale $locale, ?string $domainUID) {
        // kd pro kterou zemi se ma brat
        $countryCode = mb_strtolower($locale->getCountry());
        
        $conn = $this->dbConnection;
        
        try {
            if( $methodName == NULL) { // neni zadano method name, ma se volat nejaky kontroller
                $sql = 'SELECT * FROM '. $this->structureTablePreffix . $countryCode .' where one_way = 0 and  enabled = 1 and controller_uid = :controllerId order by routes_order asc';
                $statement = $conn->prepare($sql);
                $statement->bindParam(':controllerId', $controllerId);
            } else {
                $sql = 'SELECT * FROM '. $this->structureTablePreffix . $countryCode .' where one_way = 0 and  enabled = 1 and controller_uid = :controllerId and method_name = :methodName order by routes_order asc';
                $statement = $conn->prepare($sql);
                $statement->bindParam(':controllerId', $controllerId);
                $statement->bindParam(":methodName", $methodName);
            }
            $statement->execute();
        } catch(PDOException $ex) {
            $message = "Při vyhledávání url pro controller $controllerId došlo k chybě.";
            $this->log->error($message, ["exception" => $ex]);
            throw new Exception($message, 0, $ex);
        }
        
        // projde vsechny controllery s danym $controllerId
        while( $row = $statement->fetch(PDO::FETCH_ASSOC) ) {
            $outputURL = $row["output_url"]; // testovane url
            // parametry, ktere se maji pripojit za ? do adresy
            $appendParams = $parameters;
            
            // proved prvni match
            $matched = preg_match("#\((\?{0,1}[a-zA-Z0-9]+)\)#", $outputURL, $matches);
            if( $matched != false ) { // a navratove url obsahuje nejaky parametr
                // nalezen prvni match, zkontrolovat jestli v $parameters je dany parametr a nahradit ho
                $dontHaveMatch = false; // priznak jestli nebyl nalezen match
                do {
                    $paramName = $matches[1];
                    if( startsWith($paramName, '?') ) { // starts with ?, is not required
                        $paramName = mb_substr($paramName, 1); // remove ? from star
                        
                        if( isset($parameters[$paramName]) ) { // parameter is present
                            // replace with parameter
                            $outputURL = str_replace('(?'. $paramName .')', $parameters[$paramName], $outputURL);
                            unset($appendParams[$paramName]);
                        } else {
                            // remove parameter from output url
                            $outputURL = str_replace('(?'. $paramName .')', '', $outputURL);
                        }
                    } else {
                        if( !isset($parameters[$paramName]) ) { // param is required and $parameters do not contain $patamName, try nex record to match
                            $dontHaveMatch = true;
                            break;
                        }
                        
                        // replace with parameter
                        $outputURL = str_replace("($paramName)", $parameters[$paramName], $outputURL);
                        unset($appendParams[$paramName]);
                    }

                } while( preg_match("#\((\?{0,1}[a-zA-Z0-9]+)\)#", $outputURL, $matches) );
                if( $dontHaveMatch ) { // jit na dalsi pravidlo
                    continue;
                }
            }
            
            $appendArray = [];
            foreach($appendParams as $index => $value) {
                $appendArray[] = $index ."=". $value;
            }
            
            if( !isEmpty($appendArray) ) {
                $outputURL .= "?". implode('&', $appendArray);
            }
            return $outputURL;
            
        }
        
        return NULL;
    }

    /**
     * @param ConnectionInterface $conn db connection
     * @param array $row row from structure_xx
     * @return PageInfo informace o strance ziskane z db
     */
    protected function createPageInfo(ConnectionInterface $conn, array $row, $url, $countryCode) {
        $pageInfo = new PageInfo();
        $pageInfo->setControllerId($row["controller_uid"]);
        $pageInfo->setTitle($row["title"]);
        $pageInfo->setShortTitle($row["short_title"]);
        $pageInfo->setDesctiprion($row["description"]);
        $pageInfo->setKeywords($row["keywords"]);
        $pageInfo->setRobots($row["robots"]);
        $pageInfo->setURL($url);
        $pageInfo->setMethodName($row["method_name"]);
        
        // prochazi predky ke korenu
        $breadcrumbs = [];
        
        $parentId = $row["parent_id"];
        $sql = 'SELECT * FROM '. $this->structureTablePreffix . $countryCode .' where id = :parentId';
        $statement = $conn->prepare($sql);
        $statement->bindParam(':parentId', $parentId);
        $statement->execute();
        
        // FIXME: musi prochazet i do hloubky aby se dostal az ke korenu
        while( $bcRow = $statement->fetch(PDO::FETCH_ASSOC) ) {
            $bcPageInfo = new PageInfo();
            $bcPageInfo->setControllerId($bcRow["controller_uid"]);
            $bcPageInfo->setTitle($bcRow["title"]);
            $bcPageInfo->setShortTitle($bcRow["short_title"]);
            $bcPageInfo->setDesctiprion($bcRow["description"]);
            $bcPageInfo->setKeywords($bcRow["keywords"]);
            $bcPageInfo->setRobots($bcRow["robots"]);
            
            $breadcrumbs[] = $bcPageInfo;
            
            $parentId = $bcRow["parent_id"];
            $statement->closeCursor(); // uvolnit cursot
            $statement->bindParam(':parentId', $parentId);
            $statement->execute();
        }
        
        $breadcrumbs = array_reverse($breadcrumbs);
        $pageInfo->setBreadcrumbs($breadcrumbs);
        
        return $pageInfo;
    }
    
    /**
     * nastavuje interceptory platne pro vsechny controllery z teto route definition
     * @param array $interceptors
     */
    function setInterceptors(array $interceptors) {
        $this->interceptors = $interceptors;
        // pridava PageInfoInterceptor ktery nacit informace o prave zobrazovane strance
        $this->interceptors[] = new PageInfoInterceptor();
    }
    
    /**
     * nastavuje interceptory jen pro specificke controllery
     * @param array $interceptors pole poli kde klicem je identifier controlleru a hodnota je pole interceptoru, ktere se na nem maji provezt
     */
    function setControllerInterceptors(array $interceptors) {
        $this->controllerInterceptors = $interceptors;
    }
    
    /**
     * adds ides of interceptors to be called around $controllerId
     * @param string $controllerId
     * @param string[] $interceptorIdes
     * @throws Exception
     */
    function addControllerInterceptors($controllerId, array $interceptorIdes) {
        if( array_key_exists($controllerId, $this->controllerInterceptors) ) {
            throw new Exception("Interceptors for controller $controllerId already added.");
        }
        $this->controllerInterceptors[$controllerId] = $interceptorIdes;
    }
}
