<?php

namespace IZON\MVC;

use Exception;
use IZON\DI\Container;
use IZON\Logs\Logger;
use IZON\MVC\Context\Context;
use IZON\MVC\Context\WebContext;
use IZON\MVC\Exceptions\Handlers\AppExceptionHandler;
use IZON\MVC\Exceptions\Handlers\IAppExceptionHandler;
use IZON\MVC\Exceptions\Handlers\InternalServerErrorExceptionHandler;
use IZON\MVC\Exceptions\InternalServerErrorException;
use IZON\MVC\Factories\ServerResponseFactory;
use IZON\MVC\Interceptors\CompatibleWrapperInterceptor;
use IZON\MVC\Interceptors\Interceptor;
use IZON\MVC\Interceptors\PreHandleResult;
use IZON\MVC\Locale\LocaleResolver;
use IZON\MVC\Messages\HttpRequest;
use IZON\MVC\Messages\HttpRequestInterface;
use IZON\MVC\Messages\HttpResponseInterface;
use IZON\MVC\Messages\ResponseEmitter;
use IZON\MVC\Routers\Router;
use IZON\MVC\URLDomain\URLDomainResolverInterface;
use IZON\MVC\Views\View;
use Psr\Container\ContainerInterface;
use ReflectionMethod;
use ReflectionParameter;
use Throwable;
use function GuzzleHttp\Psr7\stream_for;
use function IZON\String\startsWith;
use function mb_strpos;
use function mb_substr;

/**
 * aplikace
 */
class App {

    //region PROPERTIES

    /**
     * v jakem adresari se nachazi cela aplikace
     * @var string
     */
    protected $baseDir;

    /**
     * konfugurace aplikace
     * @var Config
     */
    protected $config;

    /**
     * logger pro logovani prubeho vykonavani aplikace
     * @var Logger $log
     */
    protected $log;

    /**
     * DI container z ktereho je mozne vytahovat controllery
     * @var Container|null
     */
    protected $container;

    /**
     * router, ktery podle url vrati ktery controller se ma zavoloat
     * @var Router
     */
    protected $urlRouter;

    /**
     * zjisti jake locale se ma pouzit
     * @var LocaleResolver
     */
    protected $localeResolver;
    /**
     * @var URLDomainResolverInterface
     */
    protected $domainResolver;

    /**
     * slouzi k ziskani view, ktere se ma vyrenderovat
     * @var array
     */
    protected $viewResolvers;

    /**
     * informace jestli se pouziva modrewrite
     * @var boolean
     */
    protected $modRewrite = true;

    /**
     * resolvery pro zpracovani vyjimek vyhozenych systemech
     * @var IAppExceptionHandler[]
     */
    protected $exceptionHandlers = [];

    /**
     * relative Path to file that require on init app error
     * @var string
     */
    protected $initErrorPage = 'errors/500.php';

    //endregion

    /**
     * App constructor.
     * @param Config $config
     */
    public function __construct(Config $config) {
        $this->baseDir = $config->getBaseDir();
        $this->config = $config;

        define("__BASE_DIR__", $config->getBaseDir());
    }

    /**
     * provede inicializaci aplikace
     */
    public function init() {
        try {
            // init
            $this->config->init();

            // inicializuje LocaleResolver
            // TODO: vyresit najak univerzalneji
            $this->localeResolver = $this->config->getLocaleResolver();

            $this->domainResolver = $this->config->getDomainResolver();
            // nastavuje viewResolvery pro vytvareni vire
            $this->viewResolvers = $this->config->getViewResolvers();
            // get exception handlers
            $this->exceptionHandlers = $this->config->getExceptionHandlers();

            // nastavuje logger
            $this->log = $this->config->getAppLogger();

            // nastavuje container
            $this->container = $this->config->getContainer();

            // pokud je definovan nejaky uzivarelsky session handler pouzij ho
            if ($this->container->has('session.sessionHandler')) {
                //                ini_set("session.gc_probability", 1);
                //                ini_set("session.gc_divisor", 1);
                $sessionHandler = $this->container->get('session.sessionHandler');
                session_set_save_handler($sessionHandler);
            }

            $this->urlRouter = $this->config->getRouter();
        } catch (Throwable $exception) {
            $handled = false;

            // udelat dump chyby
            // !!!!!!! NEZAKOMENTOVAVAT TENTO IF JINAK SE PRI CHYBLE PRIPOJENI NA OSTRYM SERVERU DUMPUJE HESLO NA DB !!!!!!!!!!!!!!!!
            if ($this->config->isDevelServer()) { // isDevelServer is defaulted to false
                var_dump($exception);
            }
            if ($this->config->getAppLogger() instanceof Logger) {
                // nastavi logger protoze nemusibyt vubec nastaven
                $this->log = $this->config->getAppLogger();
                $this->log->error("Při inicializaci aplikace doslo k chybe", ["exception" => $exception]);
            }

            if (!empty($this->config->getExceptionHandlers())) {
                // if i have handlers to handle exception
                $this->exceptionHandlers = $this->config->getExceptionHandlers();
                $handled = $this->handleExceptionByHandlers($exception, null, null);
            }

            if (!$handled){
                $defaultHandler = new InternalServerErrorExceptionHandler($this->initErrorPage);
                $defaultHandler->handleException($exception, null);
            }
            die;
        }
    }

    /**
     * spusti zpracocani dotazu db dotazu
     */
    public function dispatch() {
        try {
            // vytvori a nastavi request
            $httpRequest = $this->populateHTTPRequest($this->container);
            // vytvori session id a posle ho prohlizeci, potom session odemce
            $httpRequest->getSession()->init();

            // TODO: asi vydejit obsluhu presmerovani nekam ma jedno misto
            // najde routovani
            $routeInfo = $this->urlRouter->findRoute($httpRequest);
            if($routeInfo->getRedirectCode() !== null) {
                // do redirect
                $this->log->info("redirect from router to " . $routeInfo->getURL());
                header("Location: " . $routeInfo->getURL());
                // ukoncit vykonavani
                return;
            }

            // add new URL query parameter to query parameters
            $queryParameters = $routeInfo->getParameters();
            foreach($httpRequest->getQueryParams() as $key => $value) {
                $queryParameters[$key] = $value;
            }
            $httpRequest = $httpRequest->withQueryParams($queryParameters);

            $this->log->info("Routing url " . $httpRequest->getURL() . " to " . $routeInfo->getContrlollerId());

            // najde interceptory
            $interceptors = $routeInfo->getInterceptors();
            // if interceptor specified by identifier, replace by interceptor from container
            for ($i = 0; $i < count($interceptors); $i++) {
                $interceptor = $interceptors[$i];
                if (is_string($interceptor)) {
                    $this->log->info("Reatriving interceptors " . $interceptor . " from container");
                    $interceptor = $this->container->get($interceptor);
                    $interceptors[$i] = $interceptor;
                }
                if(is_subclass_of($interceptor, Interceptor::class)) {
                    $interceptor = new CompatibleWrapperInterceptor($interceptor);
                    $interceptors[$i] = $interceptor;
                }
            }

            $this->log->info("Retrieved interceptors for controller " . $routeInfo->getContrlollerId() . " from container");

            // zavola preHandle pro interceptory
            foreach ($interceptors as $interceptor) {
                $this->log->info("Calling preHandle of interceptor " . get_class($interceptor) . "");
                $httpRequest = $interceptor->preHandle($httpRequest);
                $preHandleResult = $httpRequest->getAttribute(PreHandleResult::class);
                if ($preHandleResult instanceof PreHandleResult) { // ma se provect presmerovani
                    $redirectURL = $preHandleResult->getRedirectAddress();
                    if ($preHandleResult->isInternalRedirect()) {
                        $redirectControlerId = $preHandleResult->getControllerId();
                        // najde url pro presmerovani
                        $methodName = $preHandleResult->getMethodName();
                        $locale = $httpRequest->getLocale();
                        if ($preHandleResult->getLocale() != NULL) {
                            $locale = $preHandleResult->getLocale();
                        }
                        $redirectURL = $this->urlRouter->findURL($redirectControlerId, $preHandleResult->getParameters(), $preHandleResult->getMethodName(), $locale);
                    }

                    // do redirect
                    $this->log->info("redirect to " . $redirectURL);
                    header("Location: " . $redirectURL);
                    // terminate script execution
                    return;
                }
            }

            $controller = $this->container->get($routeInfo->getContrlollerId());

            $this->log->info("Retrieved controller " . get_class($controller) . " from container");

            // vyvolat metodu controlleru
            $methodName = $routeInfo->getMethodName();
            if ($methodName === NULL) { // $methodName not specified
                $methodName = "execute"; //  default to execute
            }

            if (!method_exists($controller, $methodName)) {
                $errorMessage = "Retrieved controller " . get_class($controller) . " doesn't have requested method " . $methodName;
                $this->log->error($errorMessage);
                throw new Exception($errorMessage);
            }
            $factory = new ServerResponseFactory();
            $response = $factory->createResponse();

            $calledMethodReflection = new ReflectionMethod($controller, $methodName);
            /* @var $methodParamDefs ReflectionParameter[] */
            $methodParamDefs = $calledMethodReflection->getParameters();

            $methodParameters = [];
            foreach($methodParamDefs as $key => $paramDef) {
                if(
                    $paramDef->getClass()->name == HttpRequest::class
                    || $paramDef->getClass()->name == HttpRequestInterface::class
                    || $paramDef->getClass()->isSubclassOf(HttpRequestInterface::class)
                ) {
                    $methodParameters[] = $httpRequest;
                } else if(
                    $paramDef->getClass()->name == Context::class
                    || $paramDef->getClass()->isSubclassOf(Context::class)
                ) {
                    if($routeInfo->getContext() === NULL) {
                        throw new InternalServerErrorException(
                            "Conntroller method "
                            .$methodParamDefs[1]->getClass()->name
                            ."::"
                            .$calledMethodReflection->name
                            ." expects second parameter to be Context but none gven."
                        );
                    }
                    $methodParameters[] = $routeInfo->getContext();
                }else if(
                    $paramDef->getClass()->name == HttpResponseInterface::class
                    || $paramDef->getClass()->isSubclassOf(HttpResponseInterface::class)
                ) {
                    $methodParameters[] = $response;
                }
            }

            // call controller method

            $returned = $calledMethodReflection->invokeArgs($controller, $methodParameters);
            if(is_a($returned, ModelAndView::class)) {
                $response = $response->withModelAndView($returned);
            } else {
                $response = $returned;
            }
            // zavola postHandle pro interceptory
            foreach ($interceptors as $interceptor) {
                $response = $interceptor->postHandle($httpRequest, $response);
            }
            $modelAndView = $response->getModelAndView();
            if ($modelAndView != NULL && !$modelAndView->isRedirect()) { // a metoda neco vraci
                /** @var View $view */
                $view = $this->createAndPopulateView($routeInfo->getContext(), $modelAndView);

               $render =  $view->render(true);
               $response = $response->withBody(stream_for($render))->withHeader('Content-Type', $this->getHeaderContentType());
            } elseif($modelAndView != NULL) {
                $redirectControlerId = $modelAndView->getViewName();

                if ($modelAndView->getRedirectType() == ModelAndView::REDIRECT_TYPE_URL) { // presmerovat na url
                    $redirectURL = $redirectControlerId;
                } else if ($modelAndView->getRedirectType() == ModelAndView::REDIRECT_TYPE_CONTROLLER) { // presmerovat na controller
                    // najde url pro presmerovani
                    $locale = $httpRequest->getLocale();
                    if ($modelAndView->getLocale() != NULL) {
                        $locale = $modelAndView->getLocale();
                    }
                    $redirectURL = $this->urlRouter->findURL($redirectControlerId, $modelAndView->getRedirectParameters(), $modelAndView->getMethodName(), $locale);
                } else {
                    throw new Exception("Nepodporovany typ presmerovani " . $modelAndView->getRedirectType());
                }

                // do redirect
                $this->log->info("redirect to " . $redirectURL);

                if ($modelAndView->isPermanentRedirect()) { // is permanent redirect
                    $response = $response->withStatus(301);
                }
                $response = $response->withHeader('location', $redirectURL);
            }
            $emitter = new ResponseEmitter();
            $emitter->emit($response);
        } catch (\Throwable $exception) {
            if ($this->container->has("develserver") && $this->container->get("develserver")) {
                var_dump($exception);
            }

            // handle exception
            try {
                $result = False;
                if(!empty($httpRequest)) {
                    if(empty($routeInfo)) {
                        $pageInfo = new PageInfo();
                        $pageInfo->setURL($httpRequest->getURL());
                        $httpRequest->addParameter(PageInfo::PAGE_INFO_INDENTIFIER, $pageInfo);
                        $context = new WebContext($httpRequest->getSession(), $pageInfo);
                        $context->setLocale($httpRequest->getLocale());
                        $context->setRouter($this->urlRouter);
                        $result = $this->handleExceptionByHandlers($exception, $httpRequest, $response, $context);
                    } else {
                        $result = $this->handleExceptionByHandlers($exception, $httpRequest, $response, $routeInfo->getContext());
                    }
                }
                if(!$result){
                    //exception was not handled
                    $defaultHandler = new InternalServerErrorExceptionHandler($this->initErrorPage);
                    $defaultHandler->handleException($exception, null);
                }
            } catch(Throwable $exception) {
                $this->config->getAppLogger()->error('Handle of exception fail', ['exception' => $exception]);

                if($this->container->has("develserver") && $this->container->get("develserver")) {
                    var_dump($exception);
                    throw $exception;
                }
            }

            //end application
            return;
            // TODO: asi provezt uvolneni resourcu
        }
   }

    /**
     * @param Throwable $exception
     * @param HttpRequest|null $httpRequest
     * @param Context|null $context
     * @return bool success of handle exception
     * @throws Exception handle exception fail
     */
    protected function handleExceptionByHandlers(Throwable $exception,?HttpRequestInterface $httpRequest, ?HttpResponseInterface $response = null, ?Context $context = null) {
        //go through handlers and find suitable handler
        if(empty($response)) {
            $factory = new ServerResponseFactory();
            $response = $factory->createResponse();
        }
        foreach($this->exceptionHandlers as $exceptionHandler) {
            if(!$exceptionHandler->isFor($exception)) {
                continue;
            }
            $this->log->log(
                $exceptionHandler->getLogLevel(),
                "Request to url "
                .($httpRequest !== NULL ? $httpRequest->getURL() : ' null ')
                ." faild [user-agent='"
                .filter_input(INPUT_SERVER, 'HTTP_USER_AGENT')
                ."', referer='"
                .filter_input(INPUT_SERVER, 'HTTP_REFERER')
                ."', sessionId='"
                .session_id()
                ."']",
                ['exception' => $exception]
            );
            $response = $exceptionHandler->handleException($exception, $httpRequest, $response);

            if($response != NULL && $response->getModelAndView() != null) {
                $modelAndView = $response->getModelAndView();
                if(!$modelAndView->isRedirect()) {
                    $view = $this->createAndPopulateView($context, $modelAndView);
                    $render =  $view->render(true);
                    $response = $response->withBody(stream_for($render))->withHeader('Content-Type', $this->getHeaderContentType());
                } else {
                    throw new Exception('Handle exception can not be redirected');
                }
            }
            $emitter = new ResponseEmitter();
            $emitter->emit($response);
            return true;
        }
        $this->log->error(
            "Not found exception handler on url :".$httpRequest->getURL()." ",
            ['exception' => $exception]
        );
        return false;
    }


    protected function createAndPopulateView(Context $context, ModelAndView $modelAndView) {
        $this->log->info("Searching for view with name " . $modelAndView->getViewName());

        $view = NULL;
        foreach ($this->viewResolvers as $viewResolver) {
            /** @var View $viewResolver */
            $view = $viewResolver->buildView($modelAndView->getViewName()); // try to create view
            if ($view !== NULL) {
                break;
            }
        }

        if ($view == NULL) {
            throw new Exception("Nenalezeno potrebne view '{$modelAndView->getViewName()}'.");
        }

        // provede inicializaci view
        $view->init($context, $modelAndView->getParameters());

        return $view;
    }

    /**
     * Zrapcovava http request do podoby zpracovavane dale
     *
     *
     * @return HttpRequest
     * @throws Exception
     */
    protected function populateHTTPRequest(ContainerInterface $container) {
        $method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
        $uri = HttpRequest::getUriFromGlobals();

        $serverRequestFactory = new Factories\ServerRequestFactory();

        /* @var $serverRequest HttpRequest */
        $serverRequest = $serverRequestFactory->createServerRequest($method, $uri, $_SERVER);

        // filter_input ne vzdy funguje pro pouziti na  INPUT_SERVER
        //        $requestURL = filter_input(INPUT_SERVER, 'REQUEST_URI');
        $requestURL = $_SERVER['REQUEST_URI'];
        $domainUID = $this->domainResolver->resolveCurrentDomain();
        $domain = $this->domainResolver->resolveDomainName($domainUID);
        // najdi ? a odstran vse za nim, tj. parametry url
        $position = mb_strpos($requestURL, '?');
        if ($position !== false) {
            $requestURL = mb_substr($requestURL, 0, $position);
        }


        $locale = $this->localeResolver->resolveLocale($domainUID,$requestURL, $_REQUEST);
        $modifiedRequestURL = $this->localeResolver->modifyRequestURL($domainUID, $requestURL, $locale);

        // FIXME: passed incorrectly, $serverRequest should be immutable
        $serverRequest = $serverRequest
            ->withLocale($locale)
            ->withSession(new \IZON\MVC\Session\HttpSession());
        $serverRequest->setURL($modifiedRequestURL);
        $serverRequest = $serverRequest->withDomain($domain)->withDomainUID($domainUID);


//

        return $serverRequest;
    }

    protected function getBaseDir() {
        return $this->baseDir;
    }

    /**
     * @return string
     */
    protected function getHeaderContentType(): string {
        $mimeType = 'text/html';
        $headerList = headers_list();
        foreach($headerList as $header) {
            if(startsWith(strtolower($header), 'content-type')) {
                $mimeType = str_replace('Content-type: ', '', $header);
                break;
            }
        }
        return "{$mimeType}; charset=utf-8";
    }
}
