<?php


namespace IZON\MVC\Routers;


use FastRoute\RouteParser\Std;
use IZON\Lang\NotImplementedException;
use IZON\MVC\Messages\HttpRequestInterface;
use IZON\MVC\Routers\Helpers\Route;
use IZON\Utils\Locale;
use OpenApi\Analysis;
use OpenApi\StaticAnalyser;
use Psr\Container\ContainerInterface;
use function IZON\String\startsWith;
use const OpenApi\Annotations\UNDEFINED;


/**
 * api for writion: http://api.test.writion-2019.develop.izon.cz/v1/docs/swagger/
 * php open api: https://zircote.github.io/swagger-php/Getting-started.html
 * open api specification: https://swagger.io/docs/specification/basic-structure/
 */
class ApiRouteDefinition implements RouteDefinition {

    /**
     * @var ContainerInterface
     */
    protected $container;

    /**
     * @var string
     */
    protected $apiPrefix;

    /**
     * @var string[]
     */
    protected $endpointUIDs;

    /**
     * @var Std
     */
    protected $routeParser;

    /**
     * @var string[]
     */
    protected $routingTable = null;

    /**
     * ApiRouteDefinition constructor.
     * @param ContainerInterface $container
     * @param string[] $endpointUIDs
     */
    public function __construct(ContainerInterface $container, string $apiPrefix, array $endpointUIDs) {
        $this->container = $container;
        $this->apiPrefix = $apiPrefix;
        $this->endpointUIDs = $endpointUIDs;

        $this->routeParser = new Std();
    }

    public function findRoute(HttpRequestInterface $request) {
        if( !startsWith($request->getUri()->getPath(), $this->apiPrefix) ) {
            return null;
        }
        $this->initRoutingTable();

        $endpointPath = mb_substr($request->getUri()->getPath(), mb_strlen($this->apiPrefix));
        $httpMethod = mb_strtolower($request->getMethod());

        foreach($this->routingTable as $routeInfo) {
            $match = $routeInfo['route']->match($endpointPath);
            if( $match[0] && array_key_exists($httpMethod, $routeInfo['METHODS']) ) {
                $methodRouteInfo = $routeInfo['METHODS'][$httpMethod]['route'];

                $params = $match[1];
                foreach($params as $key => $value) {
                    if(is_numeric(
                        $key
                    )) { // numeric values represent index of match, we dont need them as we bind reqex values to names
                        unset($params[$key]);
                    }
                }

                $routeInfo = new RouteInfo();
                $routeInfo->setURL($request->getUri()->getPath());
                $routeInfo->setContrlollerId($methodRouteInfo->getControllerUID());
                if($methodRouteInfo->getMethodName()) { // ma se posilat na jinou nez defaultni metodu
                    $routeInfo->setMethodName($methodRouteInfo->getMethodName());
                }
                $routeInfo->setParameters($params);
                return $routeInfo;
            }
        }

        return null;
    }

    public function findURL($controllerId, array $parameters, $methodName, Locale $locale, ?string $domainUID) {
        // no urls to be searched
    }

    protected function initRoutingTable() {
        $this->routingTable = [];
        $analyser = new StaticAnalyser();

        foreach($this->endpointUIDs as $endpointUID) {
            $endpoint = $this->container->get($endpointUID);
            $endpointClass = new \ReflectionClass($endpoint);
            $endpointName = $endpointClass->getFileName();
            $analysis = $analyser->fromFile($endpointName);
            // Post processing
            $analysis->process(Analysis::processors());

            foreach( $analysis->openapi->paths as $pathItem) {
                $path = $pathItem->path;

                $routeParams = [
                    'path' => $path,
                    'controllerUID' => $endpointUID,
                    'routeData' => $this->routeParser->parse($path),
                ];
                $pathElement = [
                    'route' => new Route($routeParams),
                ];
                if( array_key_exists($path, $this->routingTable) ) {
                    $pathElement = $this->routingTable[$path];
                }

                foreach(self::SUPPORTED_HTTP_METHODS as $supportedMethod) {
                    $methodItem = $pathItem->$supportedMethod;
                    if( $methodItem != UNDEFINED ) {
                        $classMethod = $methodItem->_context->method;
                        $pathElement['METHODS'][$supportedMethod] = [
                            'route' => new Route( array_merge($routeParams, ['methodName' => $classMethod]) ),
                        ];
                    }
                }

                $this->routingTable[$path] = $pathElement;
            }
        }
    }

    const SUPPORTED_HTTP_METHODS = [
        'get',
        'post',
        'put',
        'delete',
        'patch',
        'trace',
        'head',
        'options',
    ];
}
