<?php


namespace IZON\MVC\Messages;


use Exception;
use Psr\Http\Message\StreamInterface;
use Traversable;

class JsonStream implements StreamInterface, \ArrayAccess, \IteratorAggregate {

    protected $data = [];

    protected $position = 0;

    /**
     * JsonStream constructor.
     * @param array|iterable|\JsonSerializable $data
     */
    public function __construct($data = []) {
        $this->setData($data);
    }

    public function getData(): array {
        return $this->data;
    }

    /**
     * @param array|iterable|\JsonSerializable $data
     */
    public function setData($data) {
        if(is_iterable($data) && !is_array($data)) {
            $data = iterator_to_array($data);
        } else if($data instanceof \JsonSerializable) {
            $json = \json_encode($data);
            $data = \json_decode($json);
        }
        $this->data = $data;
    }

    protected function getJson(): string {
        return \json_encode($this->data);
    }

    //region IteratorAggregate

    public function getIterator() {
        yield from $this->data;
    }

    //endregion

    //region ArrayAccess

    public function offsetExists($offset) {
        return array_key_exists($offset, $this->data);
    }

    public function offsetGet($offset) {
        if(!$this->offsetExists($offset)) {
            return null;
        }
        return $this->data[$offset];
    }

    public function offsetSet($offset, $value) {
        return $this->data[$offset] = $value;
    }

    public function offsetUnset($offset) {
        if(!$this->offsetExists($offset)) {
            return;
        }
        unset($this->data[$offset]);
    }

    //endregion



    public function __toString() {
        return $this->getJson();
    }

    public function close() {
        // DO NOTHING
    }

    public function detach() {
        return null;
    }

    public function getSize() {
        return strlen($this->getJson());
    }

    public function tell() {
        return $this->position;
    }

    public function eof() {
        return $this->position >= $this->getSize() - 1;
    }

    public function isSeekable() {
        return true;
    }

    public function seek($offset, $whence = SEEK_SET) {
        if($whence == SEEK_CUR) {
            $offset = $this->position + $offset;
        } else if($whence == SEEK_END) {
            $offset = $this->getSize() - 1 + $offset;
        }

        if($offset >= $this->getSize() || $offset < 0) {
            throw new \RuntimeException("Position {$offset} out of range");
        }

        $this->position = $offset;
    }

    public function rewind() {
        $this->position = 0;
    }

    public function isWritable() {
        return false;
    }

    public function write($string) {
        throw new \RuntimeException("Stream is not writable");
    }

    public function isReadable() {
        return true;
    }

    public function read($length) {
        $data = substr($this->getJson(), $this->position, $length);
        $newPosition = $this->position + $length;
        if($newPosition < $this->getSize()) {
            $this->seek($newPosition);
        }
        return $data;
    }

    public function getContents() {
        $this->getJson();
    }

    public function getMetadata($key = null) {
        return null;
    }

}
