<?php

namespace IZON\MVC\Views\Resolvers;

use IZON\Logs\Logger;
use IZON\MVC\Views\PHPView;
use IZON\MVC\Views\Resolvers\ViewResolver;
use IZON\MVC\Views\StaticContent\FilesMinificationInfo;
use IZON\MVC\Views\StaticContent\MifiableStaticContentInterface;

/**
 * resolves what view to use for rendering of admin
 * all admin views must start with admin/ prefix to be processed by this view resolver
 *
 * @author IZON s.r.o. <info@izon.cz>
 * @package IZON\MVC\Views
 */
class PHPViewResolver implements ViewResolver, MifiableStaticContentInterface {

    /**
     * path relative each moduleDir
     */
    const DEFAULT_LIBRARIES_DIR = "_libs";

    /**
     * @var string directory to search for views
     */
    protected $viewsDir;

    /**
     * @var string[] dirs containing libraries for use in view
     */
    protected $librariesDirs;

    /**
     * @var string[] files containing libraries used for rendering the view
     */
    protected $librariesFiles = [];

    /**
     * @var array array of patterns for vews that require output buffer to be disabled
     */
    protected $dissableOutputBufferPatterns = [];

    /**
     * @var array key is http path to prefix files value is fs directory to search file for
     */
    protected $staticContentDirs;

    /**
     * NOTE: probbaby not desirable for css, js, ... as js could have cross domain problem
     * @var string[] prefiexs to cicle through for statis files to balance load for fifferent servers
     */
    protected $staticContentPrefixes = [""];

    /**
     * @var array[] sub dirs that contain files for minification
     */
    protected $minifiableContentSubDirs = [["subDir" => 'js', 'fileType' => 'js'], ["subDir" => 'css', 'fileType' => 'css']];

    /**
     * @var string dir to put minified content
     */
    protected $minifiedStaticContentCachePrefix = null;

    /**
     * @var string dir to put minified content
     */
    protected $minifiedStaticContentCacheDir = null;

    /**
     * @var Logger
     */
    protected $log;

    /**
     *
     * $staticContentDirs structore example:
     * 'mvc.view.phpViewResolverConfig.phpViewStaticContentDirs' => [
     *     "/" => appSubDir("/www"),
     * ],
     *
     * $config array structure:
     * 'mvc.view.phpViewResolverConfig' => [
     *      "librariesDirs" => get('mvc.view.phpViewResolverConfig.phpViewLibrariesDirs'), // directories to load php view libratries from
     *      "librariesFiles" => [], // absolute path to other php view library files
     *      "minifiedStaticContentConfig" => get('mvc.view.phpViewResolverConfig.minifiedStaticContentConfig') //
     *      "dissableOutputBufferPatterns" => ['large-output-views/*'] // pattern to disable output cache for views
     * ],
     * // folders for view rendering libraries
     * 'mvc.view.phpViewResolverConfig.phpViewLibrariesDirs' => [ appSubDir('/views/_libs') ],
     * 'mvc.view.phpViewResolverConfig.minifiedStaticContentConfig' => ['prefix' => '/minified', 'dir' => appSubDir("/www/minified")],
     *
     * @param string $viewsDir dir that contains views
     * @param array $staticContentDirs array ['httpPath' => 'absolutePathToDirSearchfilesIn', ...]
     * @param array $config
     * @throws Exception
     */
    function __construct($viewsDir, $staticContentDirs, $config = []) {
        $this->viewsDir = $viewsDir;

        // redefine libraries dirs
        if(array_key_exists("librariesDirs", $config) ) {
            $this->librariesDirs = $config["librariesDirs"];
        } else {
            $this->librariesDirs = [$this->viewsDir .'/'. self::DEFAULT_LIBRARIES_DIR];
        }

        // redefine libraties files
        if( array_key_exists("librariesFiles", $config) ) {
            $this->librariesFiles = $config["librariesFiles"];
        }

        // dissableOutputBufferPatterns
        if( array_key_exists('dissableOutputBufferPatterns', $config) ) {
            $this->dissableOutputBufferPatterns = $config['dissableOutputBufferPatterns'];
        }

        // set directories for static content
        $this->staticContentDirs = $staticContentDirs;

        // redefine prefiexes for static content
        if( array_key_exists("staticContentPrefixes", $config) ) {
            $this->staticContentPrefixes = $config["staticContentPrefixes"];
        }

        // dir to put all minified content
        if( array_key_exists("minifiedStaticContentConfig", $config) ) {
            if( !array_key_exists('prefix', $config["minifiedStaticContentConfig"])
                || !array_key_exists('dir', $config["minifiedStaticContentConfig"]) ) {
                throw new Exception("minifiedStaticContentConfig must be in format ['prefix' => 'xxxx', 'dir' => 'xxxx']");
            }
            $this->minifiedStaticContentCachePrefix = $config["minifiedStaticContentConfig"]['prefix'];
            $this->minifiedStaticContentCacheDir = $config["minifiedStaticContentConfig"]['dir'];
        }

        $this->log = Logger::getLogger(self::class);
    }


    public function buildView($viewName) {
        $this->log->info("Trying to resolve view $viewName");

        $viewTemplatePath = $this->viewsDir ."/". $viewName . PHPView::VIEW_SUFFIX;
        // nepouziva view
        if( !file_exists($viewTemplatePath) ) {
            return NULL;
        }

        $view = new PHPView($viewTemplatePath);
        $view->setLibrariesDirs($this->librariesDirs);
        $view->setLibrariesFiles($this->librariesFiles);
        $view->setStaticContentDirs($this->staticContentDirs);
        $view->setStaticContentPrefixes($this->staticContentPrefixes);
        $view->setMinifiedStaticContentCacheDir($this->minifiedStaticContentCacheDir);
        $view->setMinifiedStaticContentCachePrefix($this->minifiedStaticContentCachePrefix);

        if( $this->disableOutputBufferForView($viewName) ) {
            $view->setUseOutputBuffer(false);
        }

        return $view;
    }

    /**
     * returns info if to disable output buffer for view
     * @param string $viewName
     * @return bool
     */
    public function disableOutputBufferForView(string $viewName): bool {
        foreach($this->dissableOutputBufferPatterns as $pattern) {
            if( fnmatch($pattern, $viewName) ) {
                return true;
            }
        }
        return false;
    }

    public function getStaticContentMinificationInfo(): array {
        $forMinification = [];

        if( $this->minifiedStaticContentCacheDir === null ) {
            return [];
        }

        foreach($this->staticContentDirs as $staticContentDir) {
            $dirBaseName = basename($staticContentDir);
            foreach($this->minifiableContentSubDirs as $minifieableSubDir) {
                $subDir = $minifieableSubDir['subDir'];
                $fileType = $minifieableSubDir['fileType'];
                $files = $this->getAllDirFiles($staticContentDir, $subDir, $fileType);
                foreach($files as $file) {
                    $fileName = $file["fileName"];
                    $filePath = $file["filePath"];
                    $targetSubDir = $file["targetSubDir"];
                    $FilesMinificationInfo = new FilesMinificationInfo(
                                                    [$filePath],
                                                    $this->minifiedStaticContentCacheDir ."/". $dirBaseName ."/".  $targetSubDir,
                                                    $fileName,
                                                    $fileType
                                            );
                    $forMinification[] = $FilesMinificationInfo;
                }
            }
        }
        return $forMinification;
    }

    /**
     * returns all files of $fileType, goes recursivelly to sub dirs
     * @param string $dirPath
     * @param string $fileType
     * @return string[]
     */
    protected function getAllDirFiles(string $baseDirPath, string $dirPath, string $fileType): array {
        $files = [];
        $dirFiles = scandir($baseDirPath ."/". $dirPath);
        foreach($dirFiles as $fileName) {
            if( $fileName == '.' || $fileName == '..' ) {
                continue; // skip . and .. dirs
            }
            if( is_dir($baseDirPath ."/". $dirPath ."/". $fileName) ) {
                // get files from subdir
                $subFiles = $this->getAllDirFiles($baseDirPath, $dirPath ."/". $fileName, $fileType);
                $files = array_merge($files, $subFiles);
            } else {
                // add file of $fileType from this dir
                $fileExtension = pathinfo($fileName, PATHINFO_EXTENSION);
                if( $fileExtension == $fileType ) {
                    $files[] = [
                        "fileName" => $fileName,
                        "filePath" => $baseDirPath ."/". $dirPath ."/". $fileName,
                        "targetSubDir" => $dirPath,
                    ];
                }
            }
        }
        return $files;
    }
}
