<?php

namespace IZON\MVC\Context;

use IZON\MVC\Session\SessionInterface;
use IZON\Utils\Locale;
use IZON\MVC\FlashMessage;
use IZON\MVC\Routers\Router;
use IZON\MVC\Session\HttpSession;

/**
 * context containing information about mvc i.e. is part of web, admin ...
 *
 * @author IZON s.r.o. <info@izon.cz>
 * @copyright Copyright 2017, IZON s.r.o.
 *
 * @package IZON\MVC\Context
 */
abstract class Context implements ContextInterface {

    /**
     * @var Locale
     */
    protected $locale;

    /**
     * @var SessionInterface
     */
    private $session;

    /**
     * @var Router
     */
    protected $router;

    /**
     * @var string indentifikator volaneho controlleru, mel by nastavit router
     */
    protected $calledControllerId;

    /**
     * @var string preffix for flash messages
     */
    protected $flastMessagesPrefix = "flashMessages";


// generated constructor
    function __construct(SessionInterface $session) {
        $this->session = $session;
    }


    public function getSession(): SessionInterface {
        return $this->session;
    }

    /**
     * returns parameter stored in session under specified name
     * it locks and unlocks session
     * @param string $name
     * @return mixed
     */
    public function getSessionParameter($name) {
        $this->session->lock();
        $value = $this->session->getParameter($name);
        $this->session->unlock();
        return $value;
    }

    /**
     * saves parameter to session under specified name
     * @param string $name
     * @param mixed $value
     */
    public function setSessionParameter($name, $value) {
        $this->session->lock();
        $this->session->setParameter($name, $value);
        $this->session->unlock();
    }

    /**
     * saves parameter to session under specified name
     * @param string $name
     */
    public function removeSessionParameter($name) {
        $this->session->lock();
        $this->session->removeParameter($name);
        $this->session->unlock();
    }
    /**
     * checks if session parameter exists
     * @param string $name
     * @return boolean
     */
    public function hasSessionParameter($name) {
        $this->session->lock();
        $status = !is_null($this->session->getParameter($name));
        $this->session->unlock();
        return $status;
    }

    /**
     * pro controller s id $controllerId najde adresu pres kterou je ho mozno volat
     *
     * @param string $controllerId id controlleru
     * @param array $parameters parametry, ptere se maji predat do adresy
     * @param string|null $methodName nazev metody, ktera se ma na controlleru zavolat
     * @param Locale $locale
     * @param string|null $domainUID
     * @param bool|null $forceDomain If true, then absolute URL is always created for given domainUID.
     * @return string
     * @throws \Exception
     */
    function getControllerURL(
            string $controllerId,
            array $parameters = [],
            $methodName = NULL,
            $locale = NULL,
            $domainUID = null,
            bool $forceDomain = false
        ) {
        if( $locale === NULL ) {
            $locale = $this->locale;
        }
        return $this->router->findURL($controllerId, $parameters, $methodName, $locale, $domainUID, $forceDomain);
    }

    /**
     * adds flash message to context
     * @param string $messageText text of flash message
     * @param string|null $type type of flash message it can be obtained under
     * @param string $level [OK|WARNING|ERROR] urgency of message
     */
    public function addFlashMessage(
            string $messageText,
            string $type = null,
            $level = FlashMessage::LEVEL_OK
        ): void {
        $message = new FlashMessage($messageText, $type, $level);

        $this->session->lock();

        /** @var array[] $messages flash messages for all contexts */
        $messages = $this->getSessionParameter($this->flastMessagesPrefix);

        $contextFlashMessages = [];
        if( isset($messages[$this->getContextId()]) ) {
            $contextFlashMessages = $messages[$this->getContextId()];
        }

        if( !isset($contextFlashMessages[$message->getType()]) ) { // no message of this type present
            $contextFlashMessages[$message->getType()] = [];
        }

        // add flash message
        $contextFlashMessages[$message->getType()][] = $message;

        // update messages in comtext
        $messages[$this->getContextId()] = $contextFlashMessages;

        //
        $this->setSessionParameter($this->flastMessagesPrefix, $messages);

        $this->session->unlock();
    }

    public function hasFlashMessages(?string $type = null): bool {
        $this->session->lock();

        /** @var array[] $messages flash messages for all contexts */
        $messages = $this->getSessionParameter($this->flastMessagesPrefix);

        $contextFlashMessages = [];
        if( isset($messages[$this->getContextId()]) ) {
            $contextFlashMessages = $messages[$this->getContextId()];
        }

        $result = array_key_exists($type, $contextFlashMessages);

        $this->session->unlock();

        return $result;
    }

    /**
     * get flash messages of only specific type and removes them from context
     *
     * @param string $type
     * @return FlashMessage[]
     */
    public function popFlashMessages(string $type): array {
        $this->session->lock();

        /** @var array[] $messages flash messages for all contexts */
        $messages = $this->getSessionParameter($this->flastMessagesPrefix);

        $contextFlashMessages = [];
        if( isset($messages[$this->getContextId()]) ) {
            $contextFlashMessages = $messages[$this->getContextId()];
        }

        $result = [];
        if( array_key_exists($type, $contextFlashMessages) ) {
            $result = $contextFlashMessages[$type];
        }

        unset($contextFlashMessages[$type]);

        $messages[$this->getContextId()] = $contextFlashMessages;
        $this->setSessionParameter($this->flastMessagesPrefix, $messages);

        $this->session->unlock();

        return $result;
    }

    /**
     * returns all currently stored flash messages and removes them from context
     * @return FlashMessage[]
     */
    public function popAllFlashMessages(): array {
        $this->session->lock();
        /** @var array[] $messages flash messages for all contexts */
        $messages = $this->getSessionParameter($this->flastMessagesPrefix);

        $contextFlashMessages = [];
        if( isset($messages[$this->getContextId()]) ) {
            $contextFlashMessages = $messages[$this->getContextId()];
        }

        $result = [];
        foreach($contextFlashMessages as $contextFlashMessagesForId) {
            $result = array_merge($result, $contextFlashMessagesForId);
        }

        $messages[$this->getContextId()] = [];
        $this->setSessionParameter($this->flastMessagesPrefix, $messages);
        $this->session->unlock();

        return $result;
    }

    /**
     * id of context e.g. web, admin, ...
     * @return string returns unique id of context
     */
    protected abstract function getContextId();


/// generated getters and setters
    function getRouter() {
        return $this->router;
    }

    function setRouter(Router $router) {
        $this->router = $router;
    }

    function getCalledControllerId() {
        return $this->calledControllerId;
    }

    function setCalledControllerId($calledControllerId) {
        $this->calledControllerId = $calledControllerId;
    }

    function getLocale() {
        return $this->locale;
    }

    function setLocale(Locale $locale) {
        $this->locale = $locale;
    }
}
