<?php

/*
 * helper functions to be used in PHPView
 */

namespace PHPView;

use IZON\DBLocale\Interceptor\DBLocaleInterceptor;
use IZON\MVC\FlashMessage;
use IZON\MVC\PageInfo;
use IZON\MVC\Views\PHPLayoutView;
use IZON\MVC\Views\PHPView;
use IZON\Utils\Locale;
use IZON\Utils\Slug;


/**
 * inserts layout part to specified place of view
 * @var string $partName jak se dana cast jmenuje
 * @var boolean $required jetli musi byt primo v danem adresari sablony
 */
function insertLayoutPart($partName, $required = false) {
    /** @var PHPLayoutView $view */
    $view = PHPView::getApplicationView();
    $view->renderLayoutPart($partName, $required);
}

/**
 * returns parameter with name $paramName passed to view vrom controller in ModelAndView
 * @param string $paramName
 * @return mixed value of parameter of null if not exists
 */
function getParameter($paramName) {
    /** @var PHPView $view */
    $view = PHPView::getApplicationView();
    return $view->getParameter($paramName);
}

/**
 * returns true if $paramName was passed to view
 * @param string $paramName
 * @return boolean
 */
function hasParameter($paramName) {
    /** @var PHPView $view */
    $view = PHPView::getApplicationView();
    return $view->hasParameter($paramName);
}

/**
 * returns true if some flash messages of $type (or any messages if $type = null) are registered
 * @param string|null $type
 * @return bool
 */
function hasFlashMessages(?string $type = null): bool {
    /** @var PHPView $view */
    $view = PHPView::getApplicationView();
    return $view->getContext()->hasFlashMessages($type);
}

/**
 * returns all flash messages of provided $type and removes them from context or empty array if no flash message registered
 * @param string $type
 * @return FlashMessage[]
 */
function popFlashMessages(string $type): array {
    /** @var PHPView $view */
    $view = PHPView::getApplicationView();
    return $view->getContext()->popFlashMessages($type);
}

/**
 * returns all flash messages and removes them from context or empty array if no flash message registered
 * @return FlashMessage[]
 */
function popAllFlashMessages(): array {
    /** @var PHPView $view */
    $view = PHPView::getApplicationView();
    return $view->getContext()->popAllFlashMessages();
}

/**
 * returns url
 * @param string $relativeURL
 */
function getURL($relativeURL) {
    return "/" . $relativeURL;
}

/**
 * @deprecated
 * returns url with current timestamp
 * @param string $relativeURL
 */
function getTimeStampedURL($relativeURL) {
    return getURL($relativeURL) ."?". time();
}

/**
 * vrati odkaz odpovidajici controlleru se zadanimy parametry
 * @param string $controllerId id controlleru z DI containeru, ktery se ma zaolat
 * @param array $params parametry, potrebne k doplneni controleru
 * @param string $methodName metoda controlleru, ktera se ma zavolat
 * @param string|Locale $locale locale string used to define which locale version of web to show
 * @param string $domainUid
 */
function getControllerURL($controllerId, array $params = [], $methodName = NULL, $locale = NULL, $domainUid = NULL, bool $domainForce = false) {
    /** @var PHPView $view */
    $view = PHPView::getApplicationView();
    if($locale == NULL) {
        $locale = $view->getLocale();
    } else if(is_string($locale)) {
        $locale = Locale::forLocaleTag($locale);
    }
    //    var_dump($locale);
    $url = $view->getRouter()->findURL($controllerId, $params, $methodName, $locale, $domainUid, $domainForce);
    return str_replace("&", "&amp;", $url);
}

/**
 * formats number for better number view
 * @param float $amount
 *
 * TODO: pridat fomatovani na zaklade locale
 */
function formatNumber($amount, Locale $locale = NULL) {
    $round = 0;
    // TODO: pridat rounding
    $str = number_format($amount, $round, ',', '&nbsp;');
    return $str;
}

/**
 * returns locaje class containing language and country part
 * @return Locale
 */
function getLocale() {
    /** @var PHPView $view */
    $view = PHPView::getApplicationView();
    return $view->getLocale();
}

/**
 *
 * @return Locale
 */
function getLocaleEx() {
    return \PHPView\getParameter(DBLocaleInterceptor::AVAILABLE_AS);
}

/**
 * Render js a css
 * @deprecated use includeScript/includeScripts and includeStyle/includeStyles instead
 *
 * $data = either one dimensional field with paths to files, or<br>
 * field fields, where nested fierld has structure<br>
 * $script[external] - jedna se o externi js/css - pokud neni, netreba zadavat<br>
 * $script[src] - url, ze ktere se ma script natahnout<br>
 * pro css je jeste dostupny $script[media] - pro jaky typ media budou css pouzita - nepovinny parametr
 * <br><br>
 * For example: $data = [['external'=>true, 'src' => 'http://www.something-somewher.com/script.js']]<br>
 * $data = ['js/app.js', 'js/common.js', 'js/jquery.js']<br>
 *
 *
 * @param array $data
 * @param string $type [js|css]
 * @param string $context [web|admin]
 */
function insertScripts(array $data, $type, $context = 'web') {
    if(is_array($data) && !empty($data)) {
        foreach($data as $key => $value) {
            if(empty($value)) {
                continue;
            }
            // little bit hack for admin css and js files
            if(!is_array($value) && $context == 'admin') {
                $value = ['src' => $value];
            }

            if(!is_array($value)) {
                $value = explode("?", $value)[0];
                if($type == 'js') {
                    includeScript("/" . $value);
                } elseif($type == 'css') {
                    includeStyle("/" . $value);
                }
            } else {
                if($value['external']) {
                    if($type == 'js')
                        echo '<script type="text/javascript" src="' . $value['src'] . '"></script>' . LF;
                    elseif($type == 'css')
                        echo '<link href="' . $value['src'] . '" rel="stylesheet" type="text/css"' . ($value['media'] != '' ? ' media="' . $value['media'] . '"' : '') . ' />' . LF;
                } else {
                    $src = explode("?", $value['src'])[0];
                    if($type == 'js') {
                        includeScript("/". $src);
                    } elseif($type == 'css') {
                        includeStyle("/". $src);
//                        echo '<link href="' . $urlFunction($value['src']) . '" rel="stylesheet" type="text/css"' . ($value['media'] != '' ? ' media="' . $value['media'] . '"' : '') . ' />' . LF;
                    }
                }
            }
        }
    }
}

function includeScript($scriptURL, $type = "js") {
    /** @var PHPView $view */
    $view = PHPView::getApplicationView();
    $staticFileURL = $view->getStaticFileURL($scriptURL);
    echo '<script type="text/javascript" src="'. $staticFileURL .'"></script>'. PHP_EOL;
}

function includeScripts(array $scripts, $type = "js") {
    /** @var PHPView $view */
    $view = PHPView::getApplicationView();
    foreach($scripts as $script) {
        $staticFileURL = $view->getStaticFileURL($script);
        echo '<script type="text/javascript" src="' . $staticFileURL .'"></script>'. PHP_EOL;
    }
}

function includeStyle($style, $type = "css", $params = []) {
    /** @var PHPView $view */
    $view = PHPView::getApplicationView();
    $staticFileURL = $view->getStaticFileURL($style);
    echo '<link href="'. $staticFileURL .'" rel="stylesheet" type="text/css" ' . ($params['media'] != '' ? ' media="' . $params['media'] . '"' : '') .' />'. PHP_EOL;
}

function includeStyles(array $styles, $type = "css", $params = []) {
    /** @var PHPView $view */
    $view = PHPView::getApplicationView();
    foreach($styles as $style) {
        $staticFileURL = $view->getStaticFileURL($style);
        echo '<link href="'. $staticFileURL .'" rel="stylesheet" type="text/css" ' . ($params['media'] != '' ? ' media="' . $params['media'] . '"' : '') .' />'. PHP_EOL;
    }
}

/**
 * inserts file content to view where function was called
 * @param string $filePath
 */
function insertFileContent($filePath) {
    /** @var PHPView $view */
    $view = PHPView::getApplicationView();
    $staticFileURL = $view->getStaticFileURL($filePath);
    return file_get_contents($staticFileURL);
}

/**
 * Retruns \IZON\MVC\PageInfo object
 *
 * @return PageInfo
 */
function getPageInfo() {
    $pageInfo = \PHPView\getParameter(PageInfo::PAGE_INFO_INDENTIFIER);
    return $pageInfo;
}

/**
 * Check if actual page controller is same like controllerId param
 * @param string $controllerId to check
 * @return bool True - yes is actual False - not actual
 */
function isControllerActualPageController($controllerId) {
    /** @var PageInfo $pageInfo */
    $pageInfo = \PHPView\getParameter(PageInfo::PAGE_INFO_INDENTIFIER);
    if($pageInfo->getControllerId() == $controllerId) {
        return true;
    }
    return false;
}

/**
 * Clear string from diacritics and other "dangerous" characters, uses \IZON\Utils\Slug()
 *
 * @param string $string
 * @return string
 */
function clearString($string) {
    return Slug::createSlug($string);
}


/// backward compatibility
/**
 * @deprecated use PHPView\Thumber\getThumbnail
 * Return image url for source path or Image object
 * @param \IZON\IO\Image|string $sourceImage source Image Object or path to image file with first "/" path_to_file
 * @param array $parameters parameters for resizing of image, supported settings:<br>
 * width => integer, <br>height => integer, <br>type => [cover|contain], default is contain, <br>quality = number, default = 90<br>
 * for example: ["width" => 1234, "height" => 1234, "type" => "contain"]
 *
 * @return string
 */
function getThumbnail($sourceImage, array $parameters = []) {
    return Thumber\getThumbnail($sourceImage, $parameters);
}

/**
 * @deprecated use PHPView\Thumber\getThumb
 * Only wrapper for getThumbnail, for more detail see there
 * @param \IZON\IO\Image|string $sourceImage source Image Object or path to image file with first "/"
 * @param array $parameters parameters for resizing of image, supported settings:<br>
 * width => integer, <br>height => integer, <br>type => [cover|contain], default is contain, <br>quality = number, default = 90, max is 100 - create image without any compression<br>
 * for example: ["width" => 1234, "height" => 1234, "type" => "contain"]
 * @return string
 */
function getThumb($sourceImage, array $parameters = []) {
    return getThumbnail($sourceImage, $parameters);
}

/**
 * @deprecated use insertFileContent instead
 * Returns svg file content
 *
 * @param string $name svg file name
 * @param string $path path to svg file, default is htdocs/www/images/svg
 * @return string
 */
function insertSvg($name, $path = 'www/images/svg') {
    return file_get_contents(__BASE_DIR__ . '/' . $path . '/' . $name . '.svg');
}
