<?php

namespace IZON\MVC\Context;

use IZON\Utils\Locale;

use IZON\MVC\FlashMessage;

use IZON\MVC\Routers\Router;

use IZON\MVC\HttpSession;

/**
 * context containing information about mvc i.e. is part of web, admin ...
 *
 * @author IZON s.r.o. <info@izon.cz>
 * @copyright Copyright 2017, IZON s.r.o.
 *
 * @package IZON\MVC\Context
 */
abstract class Context {
    
    /**
     * @var Locale
     */
    protected $locale;
    
    /**
     * @var HttpSession
     */
    private $session;
    
    /**
     * @var Router 
     */
    protected $router;
    
    /**
     * @var string indentifikator volaneho controlleru, mel by nastavit router
     */
    protected $calledControllerId;
    
    /**
     * @var string preffix for flash messages 
     */
    protected $flastMessagesPrefix = "flashMessages";
    
    
// generated constructor
    function __construct(HttpSession $session) {
        $this->session = $session;
    }
    

    public function getSession() {
        return $this->session;
    }
    
    /**
     * returns parameter stored in session under specified name
     * it locks and unlocks session
     * @param string $name
     * @return mixed
     */
    public function getSessionParameter($name) {
        $this->session->lock();
        $value = $this->session->getParameter($name);
        $this->session->unlock();
        return $value;
    }

    /**
     * saves parameter to session under specified name
     * @param string $name
     * @param mixed $value
     */
    public function setSessionParameter($name, $value) {
        $this->session->lock();
        $this->session->setParameter($name, $value);
        $this->session->unlock();
    }
    
    /**
     * saves parameter to session under specified name
     * @param string $name
     */
    public function removeSessionParameter($name) {
        $this->session->lock();
        $this->session->removeParameter($name);
        $this->session->unlock();
    }
    /**
     * checks if session parameter exists
     * @param string $name
     * @return boolean
     */
    public function hasSessionParameter($name) {
        $this->session->lock();
        $status = !is_null($this->session->getParameter($name)); 
        $this->session->unlock();
        return $status;
    }
    
    /**
     * pro controller s id $controllerId najde adresu pres kterou je ho mozno volat
     * @param string $controllerId id controlleru
     * @param array $parameters parametry, ptere se maji predat do adresy
     * @param string $methodName nazev metody, ktera se ma na controlleru zavolat
     * @param Locale $locale
     * @return string
     * @throws Exception
     */
    function getControllerURL($controllerId, array $parameters = [], $methodName = NULL, $locale = NULL) {
        if( $locale === NULL ) {
            $locale = $this->locale;
        }
        return $this->router->findURL($controllerId, $parameters, $methodName, $locale);
    }
    
    /**
     * adds flash message to context
     * @param string $messageText text of flash message
     * @param string|null $id unique id of flash messages if not specified set default
     * @param string $level [OK|WARNING|ERROR] irgency of message
     */
    public function addFlashMessage($messageText, $id = NULL, $level = FlashMessage::LEVEL_OK) {
        $message = new FlashMessage($id, $messageText, $level);
        
        $this->session->lock();
        /** @var array[] $messages flash messages for all contexts */
        $messages = $this->getSessionParameter($this->flastMessagesPrefix);
        
        $contextFlashMessages = [];
        if( isset($messages[$this->getContextId()]) ) {
            $contextFlashMessages = $messages[$this->getContextId()];
        }
        
        if( !isset($contextFlashMessages[$message->getId()]) ) {
            $contextFlashMessages[$message->getId()] = [];
        }
        $contextFlashMessages[$message->getId()][] = $message;
        
        $messages[$this->getContextId()] = $contextFlashMessages;
        $this->setSessionParameter($this->flastMessagesPrefix, $messages);
        $this->session->unlock();
    }
    
    /**
     * get flash messages for specified 
     * @param string|null $id unique id of flash messages if not specified return all
     */
    public function getFlashMessages($id = NULL) {
        $this->session->lock();
        /** @var array[] $messages flash messages for all contexts */
        $messages = $this->getSessionParameter($this->flastMessagesPrefix);
        
        $contextFlashMessages = [];
        if( isset($messages[$this->getContextId()]) ) {
            $contextFlashMessages = $messages[$this->getContextId()];
        }
        
        $result = [];
        if( array_key_exists($id, $contextFlashMessages) ) {
            $result = $contextFlashMessages[$id];
        }
        
        unset($contextFlashMessages[$id]);
        
        $messages[$this->getContextId()] = $contextFlashMessages;
        $this->setSessionParameter($this->flastMessagesPrefix, $messages);
        $this->session->unlock();
        
        return $result;
    }
    
    /**
     * get all flash messages for context
     */
    public function getAllFlashMessages() {
        $this->session->lock();
        /** @var array[] $messages flash messages for all contexts */
        $messages = $this->getSessionParameter($this->flastMessagesPrefix);
        
        $contextFlashMessages = [];
        if( isset($messages[$this->getContextId()]) ) {
            $contextFlashMessages = $messages[$this->getContextId()];
        }
        
        $result = [];
        foreach($contextFlashMessages as $contextFlashMessagesForId) {
            $result = array_merge($result, $contextFlashMessagesForId);
        }
        
        $messages[$this->getContextId()] = [];
        $this->setSessionParameter($this->flastMessagesPrefix, $messages);
        $this->session->unlock();
        
        return $result;
    }
    
    /**
     * id of context e.g. web, admin, ... 
     * @return string returns unique id of context
     */
    public abstract function getContextId();
    
    
/// generated getters and setters
    function getRouter() {
        return $this->router;
    }

    function setRouter(Router $router) {
        $this->router = $router;
    }
    
    function getCalledControllerId() {
        return $this->calledControllerId;
    }

    function setCalledControllerId($calledControllerId) {
        $this->calledControllerId = $calledControllerId;
    }
    
    function getLocale() {
        return $this->locale;
    }

    function setLocale(Locale $locale) {
        $this->locale = $locale;
    }
}
