<?php

namespace IZON\MVC\Routers;

use Exception;

use IZON\MVC\Exceptions\PageNotFoundException;
use IZON\Utils\Locale;

use IZON\MVC\HttpRequest;
use IZON\MVC\Interceptors\PageInfoInterceptor;

use IZON\MVC\Context\AjaxContext;

/**
 * refinuje routovani pro ajaxive dotazy
 */
class AjaxRouteDefinition implements RouteDefinition {

    /**
     * interceptory vazane na jednotlive controllery
     * pole poli, klic je uid controlleru 
     * a hodnota je pole s id jednotlivych controlleru
     * @var array 
     */
    protected $controllerInterceptors = [];
    
    /**
     * interceptory pridavane pred vsechny controllery teto oute definition
     * @var array 
     */
    protected $interceptors = [];
    
    /**
     * @var string jak maji zacinat funkce, ktere se pouzivaji pro ajax 
     */
    protected $methodPrefix = "ajax";
    
    
    public function findRoute(HttpRequest $request) {
        if( !\IZON\String\startsWith($request->getURL(), "/_ajax/") ) { // neobsluhuje se timto routerem
            return NULL;
        }
        
        if( preg_match('#^/_ajax/([A-Za-z0-9.]+)/([A-Za-z0-9]+)$#', $request->getURL(), $matches) ) {  // reqexp musi byt obalen delimiterem
            $routeInfo = new RouteInfo();
            $routeInfo->setURL($request->getURL());
            $routeInfo->setContrlollerId($matches[1]);
            $routeInfo->setMethodName($matches[2]);
            $routeInfo->setParameters($request->getParameters());

            $interceptors = $this->interceptors;
            if( isset($this->controllerInterceptors[$matches[1]]) ) {
                $interceptors = array_merge($interceptors, $this->controllerInterceptors[$matches[1]]);
            }
            $routeInfo->setInterceptors($interceptors);
            
            $context = new AjaxContext($request->getSession());
            $routeInfo->setContext($context);
            
            // nastavuje id volaneho controlleru
            $request->setCalledControllerId($matches[1]);
            
            return $routeInfo;
        } else {
            throw new PageNotFoundException("Url ". $request->getURL() ." musi byt ve tvaru /_ajax/controllerId/methodName");
        }
    }

    public function findURL($controllerId, array $parameters, $methodName, Locale $locale) {
        if(\IZON\String\startsWith($methodName, $this->methodPrefix) ) { // je metoda pro obskuhu ajaxu
            $outputURL = "/_ajax/". $controllerId ."/". $methodName;
            
            $appendArray = [];
            foreach($parameters as $index => $value) {
                $appendArray[] = $index ."=". $value;
            }
            if( !\IZON\Arrays\isEmpty($appendArray) ) {
                $outputURL .= "?". implode('&', $appendArray);
            }
            return $outputURL;
            
        }
        return NULL;
    }
    
    /**
     * nastavuje interceptory jen pro specificke controllery
     * @param array $interceptors pole poli kde klicem je identifier controlleru a hodnota je pole interceptoru, ktere se na nem maji provezt
     */
    function setControllerInterceptors(array $interceptors) {
        $this->controllerInterceptors = $interceptors;
    }
    
    /**
     * nastavuje interceptory platne pro vsechny controllery z teto route definition
     * @param array $interceptors
     */
    function setInterceptors(array $interceptors) {
        $this->interceptors = $interceptors;
        // pridava PageInfoInterceptor ktery nacit informace o prave zobrazovane strance
        $this->interceptors[] = new PageInfoInterceptor();
    }
    
    /**
     * adds ides of interceptors to be called around $controllerId
     * @param string $controllerId
     * @param string[] $interceptorIdes
     * @throws Exception
     */
    function addControllerInterceptors($controllerId, array $interceptorIdes) {
        if( array_key_exists($controllerId, $this->controllerInterceptors) ) {
            throw new Exception("Interceptors for controller $controllerId already added.");
        }
        $this->controllerInterceptors[$controllerId] = $interceptorIdes;
    }
}
