<?php

namespace IZON\MVC\Routers\Helpers;

use InvalidArgumentException;
use IZON\MVC\PageInfo;

class Route {

    /**
     * @var string
     */
    protected $controlerUID;

    /**
     * @var null|RegexInfo[]
     */
    protected $regexInfos = null;

    /**
     * @var array
     */
    protected $routeData;

    /**
     * @var string
     */
    protected $title;

    /**
     * @var string|null
     */
    protected $shortTitle;

    /**
     * @var string|null
     */
    protected $methodName;

    /**
     * @var string
     */
    protected $robots;

    /**
     * @var string
     */
    protected $description;

    /**
     * @var string
     */
    protected $keywords;

    /**
     * @var mixed parent route
     */
    protected $parent;

    /**
     * Route constructor.
     * @param array $params
     */
    public function __construct(
        array $params
    ) {
        $this->controlerUID = $params['controllerUID'];
        $this->routeData = $params['routeData'];
        $this->title = $params['title'];
        $this->shortTitle = $params['shortTitle'];
        $this->methodName = $params['methodName'];
        $this->robots = $params['robots'];
        $this->description = $params['description'];
        $this->keywords = $params['keywords'];
        $this->parent = $params['parent'];
        if(!$this->isStaticRoute()) {
            $this->regexInfos = $this->buildRegexInfosForRoute($this->routeData);
        }
    }

    protected function isStaticRoute(): bool {
        return count($this->routeData) === 1 // only one
            && count($this->routeData[0]) === 1 // has only one part
            && is_string($this->routeData[0][0]); // part is string
    }

    /**
     * @param array $routeData
     * @return RegexInfo[]
     */
    private function buildRegexInfosForRoute(array $routeData): array {
        $regexInfos = [];
        foreach($routeData as $routeDataItem) {
            $regex = '';
            $variables = [];
            foreach($routeDataItem as $part) {
                if(is_string($part)) {
                    $regex .= preg_quote($part, '~');
                    continue;
                }
                [$varName, $regexPart] = $part;
                if(isset($variables[$varName])) {
                    throw new InvalidArgumentException(
                        sprintf(
                            'Cannot use the same placeholder "%s" twice',
                            $varName
                        )
                    );
                }
                if($this->regexHasCapturingGroups($regexPart)) {
                    throw new InvalidArgumentException(
                        sprintf(
                            'Regex "%s" for parameter "%s" contains a capturing group',
                            $regexPart,
                            $varName
                        )
                    );
                }
                $variables[$varName] = $varName;
                $regex .= '(?P<'.$varName.'>'.$regexPart.')';
            }
            $regexInfos[] = new RegexInfo($regex, $variables);
        }
        return $regexInfos;
    }

    private function regexHasCapturingGroups(string $regex): bool {
        if(strpos($regex, '(') === false) {
            // Needs to have at least a ( to contain a capturing group
            return false;
        }
        // Semi-accurate detection for capturing groups
        return (bool)preg_match(
            '~
                (?:
                    \(\?\(
                  | \[ [^\]\\\\]* (?: \\\\ . [^\]\\\\]* )* \]
                  | \\\\ .
                ) (*SKIP)(*FAIL) |
                \(
                (?!
                    \? (?! <(?![!=]) | P< | \' )
                  | \*
                )
            ~x',
            $regex
        );
    }

    /**
     * @param string $url
     * @return array
     */
    public function match(string $url): array {
        if($this->isStaticRoute()) {
            $matched = $url === $this->routeData[0][0];
            return [$matched, []];
        }
        foreach($this->regexInfos as $regexInfo) {
            $match = [];
            if(preg_match('#^'.$regexInfo->getRegex().'$#', $url, $match)) {
                $params = [];
                foreach($regexInfo->getVariables() as $var) {
                    $params[$var] = $match[$var];
                }
                return [true, $params];
            }
        }
        return [false, []];
    }

    public function generateURL(array $params): string {
        // FOR STATIC ROUTES
        if($this->isStaticRoute()) {
            $url = $this->routeData[0][0];
            if(!empty($params)) {
                $url .= '?'.http_build_query($params);
            }
            return $url;
        }

        // taken form   Slim Framework (https://slimframework.com)
        $segments = [];
        $segmentName = '';
        /*
         * $routes is an associative array of expressions representing a route as multiple segments
         * There is an expression for each optional parameter plus one without the optional parameters
         * The most specific is last, hence why we reverse the array before iterating over it
         */
        $expressions = array_reverse($this->routeData);
        foreach($expressions as $expression) {
            foreach($expression as $segment) {
                /*
                 * Each $segment is either a string or an array of strings
                 * containing optional parameters of an expression
                 */
                if(is_string($segment)) {
                    $segments[] = $segment;
                    continue;
                }
                /*
                 * If we don't have a data element for this segment in the provided $data
                 * we cancel testing to move onto the next expression with a less specific item
                 */
                if(!array_key_exists($segment[0], $params)) {
                    $segments = [];
                    $segmentName = $segment[0];
                    break;
                }
                $segments[] = $params[$segment[0]];
                unset($params[$segment[0]]);
            }
            /*
             * If we get to this logic block we have found all the parameters
             * for the provided $data which means we don't need to continue testing
             * less specific expressions
             */
            if(!empty($segments)) {
                break;
            }
        }
        if(empty($segments)) {
            throw new InvalidArgumentException('Missing data for URL segment: '.$segmentName);
        }
        $url = implode('', $segments);
        if(!empty($params)) {
            $url .= '?'.http_build_query($params);
        }
        return $url;
    }

    /**
     * returns true if has all $variableNames in route
     * @param array $variableNames
     * @return bool
     */
    public function hasAllVariables(array $variableNames): bool {
        if( $this->regexInfos === null // is static route
            && count($variableNames) == 0 ) { // no required parameters
            return true;
        }
        foreach($this->regexInfos as $regexInfo) {
            $hasAllVariables = true;
            $routeVariables = $regexInfo->getVariables();
            if( count($variableNames) != count($routeVariables) ) {
                break;
            }
            // test if all $variableNames are in $routeVariables
            foreach($routeVariables as $routeVariable) {
                if( !in_array($routeVariable, $variableNames) ) {
                    $hasAllVariables = false;
                    break;
                }
            }
            if( $hasAllVariables ) { // all variables found in $regexInfo
                return true;
            }
        }
        return false;
    }

    /**
     * @return string
     */
    public function getControllerUID(): string {
        return $this->controlerUID;
    }

    /**
     * @return string|null
     */
    public function getMethodName(): ?string {
        return $this->methodName;
    }

    /**
     * @return mixed
     */
    public function getParent() {
        return $this->parent;
    }

    public function getPageInfo(): PageInfo {
        $pageInfo = new PageInfo();
        $pageInfo->setControllerId($this->controlerUID);
        $pageInfo->setTitle($this->title ?: $this->shortTitle);
        $pageInfo->setShortTitle($this->shortTitle ?: $this->title);
        $pageInfo->setDescription($this->description);
        $pageInfo->setKeywords($this->keywords);
        $pageInfo->setRobots($this->shortTitle);
        return $pageInfo;
    }
}
