<?php

namespace IZON\MVC\Routers;

use \Exception;

use IZON\Utils\Locale;

use IZON\MVC\HttpRequest;
use IZON\MVC\PageInfo;
use IZON\MVC\Context\WebContext;

use IZON\MVC\Interceptors\PageInfoInterceptor;


/**
 * Definice routovani nacitana z pole
 * TODO: should be refactored
 */
class ArrayRouteDefinition implements RouteDefinition {
    
    /**
     * contains array of arrays with routing informations
     * @var array 
     */
    
    protected $defaultInverseRoutes = [];
    
    /**
     * @var array[] routing table  
     */
    protected $routes = []; 

    /**
     * interceptory pridavane pred vsechny controllery teto oute definition
     * @var array 
     */
    protected $interceptors = [];
    
    /**
     * interceptory vazane na jednotlive controllery
     * pole poli, klic je uid controlleru 
     * a hodnota je pole s id jednotlivych controlleru
     * @var array 
     */
    protected $controllerInterceptors = [];
    
    
    public function __construct(array $routingTable) {
        $this->routes = $routingTable;
        $this->interceptors = [new PageInfoInterceptor()];
    }
    
    /**
     * prida routovani specificke pro dane locale
     * @param string $locale
     * @param array $routes
     */
//    public function addLocaleRoutes($locale, $routes) {
//        $this->routes[$locale] = $routes;
//    }
    
    public function findRoute(HttpRequest $request) {
        $routes = NULL;
        // ma talove locale, routuj podle neho
        if( isset($this->routes[ mb_strtolower($request->getLocale()->getCountry()) ]) ) {
            $routes = $this->routes[mb_strtolower($request->getLocale()->getCountry())];
        } else {
            throw new Exception("No routing table for locale ". $request->getLocale()->getCountry());
        }
        
        foreach($routes as $route) {
            $urlPattern = $route['url'];
//            echo $urlPattern ." ". $request->getURL();
            $matches = [];
//            preg_match('/(?P<word>t[^s]+)/',$test,$matches);
            if( preg_match('#'. $urlPattern .'#', $request->getURL(), $matches) ) {  // reqexp musi byt obalen delimiterem
                $parameters = $matches;
                unset($parameters[0]);
                
                // prekopirovat parametry z adresy do $request
                foreach($parameters as $key => $value) {
                    if( !is_numeric($key) ) { // numeric values represent index of match, we dont need them as we bind reqex values to names
                        $request->addParameter($key, $value);
                    }
                }
                
                // pridava informace o strance ziskane z routovacich 
                $pageInfo = $this->createPageInfo($route);
                $pageInfo->setParameters($request->getParameters());
                $request->addParameter(PageInfo::PAGE_INFO_INDENTIFIER, $pageInfo);
                
                $routeInfo = new RouteInfo();
                $routeInfo->setURL($request->getURL());
                $routeInfo->setPattern($urlPattern);
                $routeInfo->setContrlollerId($route['controller_uid']);
                if( array_key_exists('method_name', $route) ) { // ma se posilat na jinou nez defaultni metodu
                    $routeInfo->setMethodName($route['method_name']);
                }
                $routeInfo->setParameters($parameters);
                
                // add context to page
                $context = new WebContext($request->getSession(), $pageInfo);
                $routeInfo->setContext($context);
                
                $interceptors = $this->interceptors;
                if( isset($this->controllerInterceptors[$route['controller_uid']]) ) {
                    $interceptors = array_merge($interceptors, $this->controllerInterceptors[$route['controller_uid']]);
                }
                $routeInfo->setInterceptors($interceptors);
                
                return $routeInfo;
            }
        }
        return NULL;
    }

    public function findURL($controllerId, array $parameters, $methodName, Locale $locale) {
        $routes = NULL;
        // ma talove locale, routuj podle neho
        if( isset($this->routes[ mb_strtolower($locale->getCountry()) ]) ) {
            $routes = $this->routes[ mb_strtolower($locale->getCountry()) ];
        } else {
            throw new Exception("No routing table for locale ". $locale->getCountry());
        }
        
        foreach($routes as $route) {
            $routeControllerId = $route['controller_uid'];
            if( $routeControllerId == $controllerId ) {
                $outputURL = $route["output_url"]; // testovane url
                //
                // parametry, ktere se maji pripojit za ? do adresy
                $appendParams = $parameters;

                // proved prvni match
                $matched = preg_match("#\((\?{0,1}[a-zA-Z0-9]+)\)#", $outputURL, $matches);
                if( $matched != false ) { // a navratove url obsahuje nejaky parametr
                    // nalezen prvni match, zkontrolovat jestli v $parameters je dany parametr a nahradit ho
                    $dontHaveMatch = false; // priznak jestli nebyl nalezen match
                    do {
                        $paramName = $matches[1];
                        if( \IZON\String\startsWith($paramName, '?') ) { // starts with ?, is not required
                            $paramName = mb_substr($paramName, 1); // remove ? from star

                            if( isset($parameters[$paramName]) ) { // parameter is present
                                // replace with parameter
                                $outputURL = str_replace('(?'. $paramName .')', $parameters[$paramName], $outputURL);
                                unset($appendParams[$paramName]);
                            } else {
                                // remove parameter from output url
                                $outputURL = str_replace('(?'. $paramName .')', '', $outputURL);
                            }
                        } else {
                            if( !isset($parameters[$paramName]) ) { // param is required and $parameters do not contain $patamName, try nex record to match
                                $dontHaveMatch = true;
                                break;
                            }

                            // replace with parameter
                            $outputURL = str_replace("($paramName)", $parameters[$paramName], $outputURL);
                            unset($appendParams[$paramName]);
                        }

                    } while( preg_match("#\((\?{0,1}[a-zA-Z0-9]+)\)#", $outputURL, $matches) );
                    if( $dontHaveMatch ) { // jit na dalsi pravidlo
                        continue;
                    }
                }

                $appendArray = [];
                foreach($appendParams as $index => $value) {
                    $appendArray[] = $index ."=". $value;
                }

                if( !\IZON\Arrays\isEmpty($appendArray) ) {
                    $outputURL .= "?". implode($appendArray, '&');
                }
                return $outputURL;
            }
        }
        
        return NULL;
    }

    /**
     * 
     * @param array $row array containing pageinfo
     * @return PageInfo informace o strance ziskane z db
     */
    protected function createPageInfo(array $row) {
        $pageInfo = new PageInfo();
        
        if(array_key_exists('title', $row) ) {
            $pageInfo->setTitle($row["title"]);
        }
        if(array_key_exists('description', $row) ) {
            $pageInfo->setDesctiprion($row["description"]);
        }
        if(array_key_exists('keywords', $row) ) {
            $pageInfo->setKeywords($row["keywords"]);
        }
        if(array_key_exists('robots', $row) ) {
            $pageInfo->setRobots($row["robots"]);
        }
                
        return $pageInfo;
    }
    
    /**
     * nastavuje interceptory platne pro vsechny controllery z teto route definition
     * @param array $interceptors
     */
    function setInterceptors(array $interceptors) {
        $this->interceptors = $interceptors;
        // pridava PageInfoInterceptor ktery nacit informace o prave zobrazovane strance
        $this->interceptors[] = new PageInfoInterceptor();
    }
    
    /**
     * nastavuje interceptory jen pro specificke controllery
     * @param array $interceptors pole poli kde klicem je identifier controlleru a hodnota je pole interceptoru, ktere se na nem maji provezt
     */
    function setControllerInterceptors(array $interceptors) {
        $this->controllerInterceptors = $interceptors;
    }
    
    /**
     * adds ides of interceptors to be called around $controllerId
     * @param string $controllerId
     * @param string[] $interceptorIdes
     * @throws Exception
     */
    function addControllerInterceptors($controllerId, array $interceptorIdes) {
        if( array_key_exists($controllerId, $this->controllerInterceptors) ) {
            throw new Exception("Interceptors for controller $controllerId already added.");
        }
        $this->controllerInterceptors[$controllerId] = $interceptorIdes;
    }
}
