<?php

namespace IZON\MVC;

use Exception;

use IZON\MVC\Context\WebContext;
use IZON\MVC\Exceptions\AccessForbiddenException;
use IZON\MVC\Exceptions\Handlers\AppExceptionHandler;
use IZON\MVC\Exceptions\Handlers\InternalServerErrorExceptionHandler;
use IZON\MVC\Exceptions\PageNotFoundException;
use ReflectionMethod;

use IZON\Logs\Logger;

use IZON\DI\Container;

use IZON\MVC\Locale\LocaleResolver;

use IZON\MVC\Interceptors\PreHandleResult;

use IZON\MVC\Routers\Router;

use IZON\MVC\HttpRequest;
use IZON\MVC\Context\Context;
use IZON\MVC\ModelAndView;

use IZON\Utils\Locale;

use IZON\MVC\Exceptions\InternalServerErrorException;

/**
 * aplikace
 */
class App {

    /**
     * v jakem adresari se nachazi cela aplikace
     * @var string
     */
    protected $baseDir;

    /**
     * jaka trida se ma pouzit pro nacitani konfiguraci
     * @var string class name
     */
    protected $configClass;

    /**
     * konfugurace aplikace
     * @var Config
     */
    protected $config;

    /**
     * logger pro logovani prubeho vykonavani aplikace
     * @var Logger $log
     */
    protected $log;

    /**
     * DI container z ktereho je mozne vytahovat controllery
     * @var Container|null
     */
    protected $container;

    /**
     * router, ktery podle url vrati ktery controller se ma zavoloat
     * @var Router
     */
    protected $urlRouter;

    /**
     * zjisti jake locale se ma pouzit
     * @var LocaleResolver
     */
    protected $localeResolver;

    /**
     * slouzi k ziskani view, ktere se ma vyrenderovat
     * @var array
     */
    protected $viewResolvers;

    /**
     * informace jestli se pouziva modrewrite
     * @var boolean
     */
    protected $modRewrite = true;

    /**
     * resolvery pro zpracovani vyjimek vyhozenych systemech
     * @var AppExceptionHandler[]
     */
    protected $exceptionHandlers = [];
    /**
     * relative Path to file that require on init app error
     * @var string
     */
    protected $initErrorPage = 'errors/500.php';

    /**
     * App constructor.
     * @param string $baseDir
     * @param string $configClass
     * @param array $parameters array of optional parameters
     *          supported array:
     *          ['initErrorPage' => 'relative path to error file '] - error in init the application
     */
    public function __construct($baseDir,
                                $configClass = Config::class, $parameters = []) {

        // sets basedir of aplication for easier url handling
        // TODO: potom nabradir necim co se jednoduseji spravuje
        /**
         * @deprecated will be removed in 4.3 use 
         */
        define("__BASE_DIR__", $baseDir);

        $this->baseDir = $baseDir;
        $this->configClass = $configClass;

        if ($parameters['initErrorPage']) {
            $this->initErrorPage = $parameters['initErrorPage'];
        }
    }

    /**
     * provede inicializaci aplikace
     */
    public function init() {
        $config = null;
        try {
            $configClass = $this->configClass;
            // vytvori konfiguraci
            /* @var $config Config */
            $config = new $configClass($this->baseDir);
            $config->init();
            $this->config = $config;

            // inicializuje LocaleResolver
            // TODO: vyresit najak univerzalneji
            $this->localeResolver = $this->config->getLocaleResolver();
            // nastavuje viewResolvery pro vytvareni vire
            $this->viewResolvers = $this->config->getViewResolvers();
            // get exception handlers
            $this->exceptionHandlers = $this->config->getExceptionHandlers();

            // nastavuje logger
            $this->log = $config->getAppLogger();

            // nastavuje container
            $this->container = $config->getContainer();

            // pokud je definovan nejaky uzivarelsky session handler pouzij ho
            if ($this->container->has('session.sessionHandler')) {
                //                ini_set("session.gc_probability", 1);
                //                ini_set("session.gc_divisor", 1);
                $sessionHandler = $this->container->get('session.sessionHandler');
                session_set_save_handler($sessionHandler);
            }

            // inicializuje routery
            $this->initRouter($config);
        } catch (\Throwable $exception) {
            $handled = false;
            if( isset($config) ) { // only test if config is created is needed, otherwise it doesn't show exeption dump
                // udelat dump chyby
                // !!!!!!! NEZAKOMENTOVAVAT TENTO IF JINAK SE PRI CHYBLE PRIPOJENI NA OSTRYM SERVERU DUMPUJE HESLO NA DB !!!!!!!!!!!!!!!!
                if ($config->isDevelServer()) { // isDevelServer is defaulted to false
                    var_dump($exception);
                }
                if ($config->getAppLogger() instanceof Logger) {
                    // nastavi logger protoze nemusibyt vubec nastaven
                    $this->log = $config->getAppLogger();
                    $this->log->error("Při inicializaci aplikace doslo k chybe", ["exception" => $exception]);
                }

                if (!empty($config->getExceptionHandlers())) {
                    // if i have handlers to handle exception
                    $this->exceptionHandlers = $config->getExceptionHandlers();
                    $handled = $this->handleExceptionByHandlers($exception, null, null);
                }
            }
            if (!$handled){
                $defaultHandler = new InternalServerErrorExceptionHandler($this->initErrorPage);
                $defaultHandler->handleException($exception, null);
            }
            die;
        }
    }

    /**
     * spusti zpracocani dotazu db dotazu
     */
    public function dispatch() {
        try {
            // vytvori a nastavi request
            $httpRequest = $this->populateHTTPRequest($this->container);
            // vytvori session id a posle ho prohlizeci, potom session odemce
            $httpRequest->getSession()->init();

            // TODO: asi vydejit obsluhu presmerovani nekam ma jedno misto
            // najde routovani
            $routeInfo = $this->urlRouter->findRoute($httpRequest);
            if ($routeInfo->getRedirectCode() !== null) {
                // do redirect
                $this->log->info("redirect from router to " . $routeInfo->getURL());
                header("Location: " . $routeInfo->getURL());
                // ukoncit vykonavani
                return;
            }

            $this->log->info("Routing url " . $httpRequest->getURL() . " to " . $routeInfo->getContrlollerId());

            // najde interceptory
            $interceptors = $routeInfo->getInterceptors();
            // if interceptor specified by identifier, replace by interceptor from container
            for ($i = 0; $i < count($interceptors); $i++) {
                $interceptor = $interceptors[$i];
                if (is_string($interceptor)) {
                    $this->log->info("Reatriving interceptors " . $interceptor . " from container");
                    $interceptors[$i] = $this->container->get($interceptor);
                }
            }

            $this->log->info("Retrieved interceptors for controller " . $routeInfo->getContrlollerId() . " from container");

            // zavola preHandle pro interceptory
            foreach ($interceptors as $interceptor) {
                $this->log->info("Calling preHandle of interceptor " . get_class($interceptor) . "");
                $preHandleResult = $interceptor->preHandle($httpRequest);

                if ($preHandleResult instanceof PreHandleResult) { // ma se provect presmerovani
                    $redirectURL = $preHandleResult->getRedirectAddress();
                    if ($preHandleResult->isInternalRedirect()) {
                        $redirectControlerId = $preHandleResult->getControllerId();
                        // najde url pro presmerovani
                        $methodName = $preHandleResult->getMethodName();
                        $locale = $httpRequest->getLocale();
                        if ($preHandleResult->getLocale() != NULL) {
                            $locale = $preHandleResult->getLocale();
                        }
                        $redirectURL = $this->urlRouter->findURL($redirectControlerId, $preHandleResult->getParameters(), $preHandleResult->getMethodName(), $locale);
                    }

                    // do redirect
                    $this->log->info("redirect to " . $redirectURL);
                    header("Location: " . $redirectURL);
                    // terminate script execution
                    return;
                }
            }

            $controller = $this->container->get($routeInfo->getContrlollerId());

            $this->log->info("Retrieved controller " . get_class($controller) . " from container");

            // vyvolat metodu controlleru
            $methodName = $routeInfo->getMethodName();
            if ($methodName === NULL) { // $methodName not specified
                $methodName = "execute"; //  default to execute
            }

            if (!method_exists($controller, $methodName)) {
                $errorMessage = "Retrieved controller " . get_class($controller) . " doestn have requested method " . $methodName;
                $this->log->error($errorMessage);
                throw new Exception($errorMessage);
            }

            $calledMethodReflection = new ReflectionMethod($controller, $methodName);
            /* @var $methodParamDefs ReflectionParameter[] */
            $methodParamDefs = $calledMethodReflection->getParameters();

            // TODO: udelat univerzalneji, aby se dalo pouzivat vice trid
            $methodParameters = [];
            if ($methodParamDefs[0]->getClass()->name == HttpRequest::class
                || $methodParamDefs[0]->getClass()->isSubclassOf(HttpRequest::class)
            ) {
                $methodParameters[] = $httpRequest;
            }
            // context is set
            if (isset($methodParamDefs[1])
                && ($methodParamDefs[1]->getClass()->name == Context::class
                    || $methodParamDefs[1]->getClass()->isSubclassOf(Context::class))
            ) {
                if ($routeInfo->getContext() === NULL) {
                    throw new InternalServerErrorException("Conntroller method " . $methodParamDefs[1]->getClass()->name . "::" . $calledMethodReflection->name . " expects second parameter to be Context but none gven.");
                }
                $methodParameters[] = $routeInfo->getContext();
            }
            // call controller method
            /* @var $modelAndView ModelAndView return value of controller */
            $modelAndView = $calledMethodReflection->invokeArgs($controller, $methodParameters);

            // zavola postHandle pro interceptory
            foreach ($interceptors as $interceptor) {
                $interceptor->postHandle($httpRequest, $modelAndView);
            }
            
            if ($modelAndView != NULL && !$modelAndView->isRedirect()) { // a metoda neco vraci
                // vytvori view
                $view = $this->createAndPopulateView($routeInfo->getContext(), $modelAndView);

                // vyrendruje view
                $view->render();
            }

            // zavola afterCompletion pro interceptory
            foreach (array_reverse($interceptors) as $interceptor) {
                $interceptor->afterCompletion($httpRequest, $modelAndView);
            }

            if ($modelAndView != NULL // mame model and view
                && $modelAndView->isRedirect() ) { // je redirect
                $redirectControlerId = $modelAndView->getViewName();

                if ($modelAndView->getRedirectType() == ModelAndView::REDIRECT_TYPE_URL) { // presmerovat na url
                    $redirectURL = $redirectControlerId;
                } else if ($modelAndView->getRedirectType() == ModelAndView::REDIRECT_TYPE_CONTROLLER) { // presmerovat na controller
                    // najde url pro presmerovani
                    $locale = $httpRequest->getLocale();
                    if ($modelAndView->getLocale() != NULL) {
                        $locale = $modelAndView->getLocale();
                    }
                    $redirectURL = $this->urlRouter->findURL($redirectControlerId, $modelAndView->getRedirectParameters(), $modelAndView->getMethodName(), $locale);
                } else {
                    throw new Exception("Nepodporovany typ presmerovani " . $modelAndView->getRedirectType());
                }

                // do redirect
                $this->log->info("redirect to " . $redirectURL);

                if ($modelAndView->isPermanentRedirect()) { // is permanent redirect
                    header("HTTP/1.1 301 Moved Permanently");
                }
                header("Location: " . $redirectURL); // send redirect
            }
        } catch (\Throwable $exception) {
            if ($this->container->has("develserver") && $this->container->get("develserver")) {
                var_dump($exception);
            }

            // handle exception
            try {
                $result = False;
                if(!empty($httpRequest)) {
                    if(empty($routeInfo)) {
                        $pageInfo = new PageInfo();
                        $pageInfo->setURL($httpRequest->getURL());
                        $httpRequest->addParameter(PageInfo::PAGE_INFO_INDENTIFIER, $pageInfo);
                        $context = new WebContext($httpRequest->getSession(), $pageInfo);
                        $context->setLocale($httpRequest->getLocale());
                        $context->setRouter($this->urlRouter);
                        $result = $this->handleExceptionByHandlers($exception, $httpRequest, $context);
                    } else {
                        $result = $this->handleExceptionByHandlers($exception, $httpRequest, $routeInfo->getContext());
                    }
                }
                if(!$result){
                    //exception was not handled
                    $defaultHandler = new InternalServerErrorExceptionHandler($this->initErrorPage);
                    $defaultHandler->handleException($exception, null);
                }
            }catch(\Throwable $exception){
                if ($this->container->has("develserver") && $this->container->get("develserver")) {
                    var_dump($exception);
                }
                $this->config->getAppLogger()->error('Handle of exception fail', ['exception' => $exception]);
            }

            //end application
            return;
            // TODO: asi provezt uvolneni resourcu
        }
   }

    /**
     * @param \Throwable $exception
     * @param \IZON\MVC\HttpRequest|null $httpRequest
     * @param Context|null $context
     * @return bool success of handle exception
     * @throws Exception handle exception fail
     */
    protected function handleExceptionByHandlers(\Throwable $exception,?HttpRequest $httpRequest,?Context $context) {
        //go through handlers and find suitable handler
        foreach ($this->exceptionHandlers as $exceptionHandler) {
            if ($exceptionHandler->isFor($exception)) {
                $this->log->log($exceptionHandler->getLogLevel(), "Request to url " . ($httpRequest !== NULL ? $httpRequest->getURL() : ' null ') . " faild [user-agent='" . filter_input(INPUT_SERVER, 'HTTP_USER_AGENT') . "', referer='" . filter_input(INPUT_SERVER, 'HTTP_REFERER') . "', sessionId='" . session_id() . "']", ['exception' => $exception]);
                $modelAndView = $exceptionHandler->handleException($exception, $httpRequest);

                if($modelAndView != NULL) {
                    if(!$modelAndView->isRedirect()) {
                        $view = $this->createAndPopulateView($context, $modelAndView);
                        $view->render();
                    } else {
                        throw new Exception('Handle exception can not be redirected');
                    }
                }
                return true;
            }
        }
        $this->log->error("Not found exception handler on url :" . $httpRequest->getURL() . " ", ['exception' => $exception]);
        return false;
    }


    protected function createAndPopulateView(Context $context, ModelAndView $modelAndView) {
        $this->log->info("Searching for view with name " . $modelAndView->getViewName());

        $view = NULL;
        foreach ($this->viewResolvers as $viewResolver) {
            /** @var Views\View $viewResolver */
            $view = $viewResolver->buildView($modelAndView->getViewName()); // try to create view
            if ($view !== NULL) {
                break;
            }
        }

        if ($view == NULL) {
            throw new Exception("Nenalezeno potrebne view");
        }

        // provede inicializaci view
        $view->init($context, $modelAndView->getParameters());

        return $view;
    }

    /**
     * initializes application router
     * @param Config $config
     */
    protected function initRouter(Config $config) {
        $this->urlRouter = $config->getRouter();
    }

    /**
     * Zrapcovava http request do podoby zpracovavane dale
     *
     *
     * @return HttpRequest
     * @throws Exception
     */
    protected function populateHTTPRequest(Container $container) {
        // filter_input ne vzdy funguje pro pouziti na  INPUT_SERVER
        //        $requestURL = filter_input(INPUT_SERVER, 'REQUEST_URI');
        $requestURL = $_SERVER['REQUEST_URI'];
        // najdi ? a odstran vse za nim, tj. parametry url
        $position = mb_strpos($requestURL, '?');
        if ($position !== false) {
            $requestURL = mb_substr($requestURL, 0, $position);
        }


        $locale = $this->localeResolver->resolveLocale($requestURL, $_REQUEST);
        $request = $this->localeResolver->modifyRequestURL($requestURL, $locale);

        $httpRequest = new HttpRequest($request, $_REQUEST, $locale);
        return $httpRequest;
    }

    protected function getBaseDir() {
        return $this->baseDir;
    }
}