<?php

namespace IZON\MVC\Views\Resolvers;

use Exception;

use IZON\Logs\Logger;

use IZON\MVC\Views\PHPLayoutView;

/**
 * trida renderujici view, resici cachovani rendrovani view, skladajici view 
 * z jednotlivych casti, ...
 * jednotliva view jsou ulozena v adresarich
 * pri devel modu naincluduje nekde dole ve strance misto na debug informace
 *
 * @author David Dryml <dryml@izon.cz>
 * @copyright Copyright 2009, HI Software s.r.o.
 *
 * @version v1.0 rc
 *
 * @package MVC
 */
class PHPLayoutViewResolver extends PHPViewResolver implements ViewResolver {
    
    /**
     * @var string containing layouts 
     */
    protected $layoutsDir;

    /**
     * @var string name of default layout to use 
     */
    protected $defaultLayoutName = "basic-layout";
    
    /**
     * @var array contains info about layouts for different views
     */
    protected $viewsLayouts = [];
    
    
    function __construct($viewsDir, $layoutsDir, $staticContentDirs, array $config = []) {
        parent::__construct($viewsDir, $staticContentDirs, $config);
        $this->layoutsDir = $layoutsDir;
        
        // redefine default layout if passed in config
        if( array_key_exists("defaultLayoutName", $config) ) { 
            $this->defaultLayoutName = $config["defaultLayoutName"];
        }
        
        // redefine other layouts if passed in config
        if( array_key_exists("viewsLayouts", $config) ) {
            $this->viewsLayouts = $config["viewsLayouts"];
        }
        
        $this->log = Logger::getLogger(self::class);
    }

    public function buildView($viewName) {
        if( !$this->viewExists($viewName) ) {
            $this->log->info("View name ". $viewName ." not hendeled by ". __CLASS__);
            return null;
        }
        
        $this->log->info("Trying to resolve view $viewName");
        
        $layoutFilePath = "";
        foreach($this->viewsLayouts as $configViewName => $config) { // test all custom layouts
            if( fnmatch($configViewName, $viewName) ) { // view name matches with config $configViewName
                $layoutFilePath = $this->layoutsDir ."/". $config["layoutName"] . PHPLayoutView::LAYOUT_SUFFIX;
                break;
            }
        }
        if( $layoutFilePath == NULL ) { // neni zmeneno nastavit defaultni
            $layoutFilePath = $this->layoutsDir ."/". $this->defaultLayoutName . PHPLayoutView::LAYOUT_SUFFIX;
        }
        
        $this->log->info("Using template $layoutFilePath");
        
        $dirsToSearchLayoutParts = $this->getDirsToSearchLayoutParts($viewName);
        
        // TODO: umoznit pouzivat i jina view
        $view = new PHPLayoutView($layoutFilePath, 
                                  $dirsToSearchLayoutParts);
        $view->setLibrariesDirs($this->librariesDirs);
        $view->setLibrariesFiles($this->librariesFiles);
        $view->setStaticContentDirs($this->staticContentDirs);
        $view->setStaticContentPrefixes($this->staticContentPrefixes);
        
        if( $this->disableOutputBufferForView($viewName) ) {
            $view->setUseOutputBuffer(false);
        }
        
        return $view;
    }
    
    /**
     * tests if set view exists
     * @param string $viewName
     * @return boolean
     */
    protected function viewExists($viewName) {
        $viewDir = $this->viewsDir ."/". $viewName;
        if( file_exists($viewDir)
            && is_dir($viewDir) ) {
            // is project specific view
            return true;
        }
        
        return false;
    }
    
    protected function getDirsToSearchLayoutParts($viewName) {
        /// find directores where to search for layout parts
        // sub dirs for viewName
        $viewNameSubDirs = $this->splitSubDirs($viewName);
        // directories where to search for layout part files
        $dirsToSearchLayoutParts = [];
        for($prefixLength = count($viewNameSubDirs); $prefixLength >= 0; $prefixLength--) {
            $viewNamePart = $this->createSubDir($viewNameSubDirs, $prefixLength);
            
            $adminViewsPath = $this->viewsDir ."/". $viewNamePart;
            if( file_exists($adminViewsPath)
                && is_dir($adminViewsPath) ) {
                 $dirsToSearchLayoutParts[] = $adminViewsPath;
            }
        }
        return $dirsToSearchLayoutParts;
    }
    
    protected function splitSubDirs($path) {
        return explode("/", $path);
    }

    /**
     * vytvoti cestu k podadresari podle polozek v poli a poctu polozek, ktere se maji\
     * pozit
     */
    protected function createSubDir($subDirsArray, $count) {
        return implode('/', array_slice($subDirsArray, 0, $count));
    }
}
